<?PHP
/**
 * class describing an attribute of an entity in an ERmodel
 * note: instances do not hold data-values but a description of the 
 * name and datatype of entity-attribute
 * 
 * @package GGF
 * @version 3.3
 * @since 3.0
 * @author Gerald Zincke, Austria
 * @copyright 2005,2006 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.html#top
 */
class GGFAttributeType {
	/** @var string the technical name of the attribute (also used in SQL tables) */
	protected $name;
	/** @var string a human readable version of the attributes name */
	protected $externalName;
		
   	/**
   	 * creates an attributetype
   	 *
   	 * Attribute types are elements of entity types 
   	 *
   	 * @param string containing the name of the attribute type, normally equal to a db column name
   	 * @param string name used for communication with the user 
   	 */
	function __construct($name, $externalName="") {
		$this->name = $name;
		$this->externalName = $externalName;
	}
	/*
	 * checks if aValue would be valid for the attribute 
	 * override this in derived classes, if needed
	 */
	public function isValid($aValue="") {
		return TRUE;
	}
	
	public function name() {
		return $this->name;
	}
	
	public function externalName() {
		if ($this->externalName == "") {
			return $this->name();
		} else {
			return $this->externalName;
		}
	}
	
	public function setExternalName($name) {
		$this->externalName = $name;
	}
	
	/*
	 * converts the value to a string that can be used as a 
	 * literal in SQL statements 
	 * must be overridden, if PHP's to string conversion is not sufficient for SQL
	 */
	public function asSQL($row) {
	}
}

class GGFNumAttribute extends GGFAttributeType {

	public function asSQL($row) {
		global $db;
		$data = $db->escape_string($row[$this->name]);
		if ($data=="") { // must not return an empty string
			$data = '0';
		} 
		return $data;
	}
}

class GGFTextAttribute extends GGFAttributeType {

	public function asSQL($row) {
		global $db;
		return "'".$db->escape_string($row[$this->name])."'";
	}
}


/**
 * class describing an entity type of an ERmodel
 * note: instances do not hold data-values but a description of the structure of entities
 * 
 * @package GGF
 * @version 4.0
 * @since 3.0
 * @author Gerald Zincke, Austria
 * @copyright 2005, 2012 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.html#top
 */
class GGFEntityType {
	protected $name;
	protected $rootTypeName="";
	protected $attribute=array();
	protected $defaultAttributeNames;
	protected $primaryKey=array();
	protected $tableNames;
	protected $joinfilter;
	protected $defaultFilter;
	protected $defaultSorter;
	
   	/**
   	 * creates a new entitytype
   	 *
   	 * @param string containing the name of the entity type, normally equall to db table name
   	 * @param array names of tables in DB, if different than entity name 
   	 * @param GGFSqlFilter  if more than one tables, condition how to join them. Note: this should not be an GGFERFilter, to avoid circular references	 
   	 */
	function __construct($name, $tableNames="", $joinfilter = "", $defaultFilter = "", $defaultSorter = "") {
		$this->name = $name;
		if ($tableNames=="") {
			$this->tableNames = array($name);
		} else {
			$this->tableNames = $tableNames;
		}
		
		if ($joinfilter == "") {
			$this->joinfilter = new GGFSqlFilter(); // initial an empty one
		} else {
			$this->joinfilter = $joinfilter;
		}
		
		if ($defaultFilter == "") {
			$this->defaultFilter = new GGFERFilter($this);
		} else {
			$this->defaultFilter = $defaultFilter;
		}
		
		if ($defaultSorter == "") {
			$this->defaultSorter = new GGFSqlSorter();
		} else {
			$this->defaultSorter = $defaultSorter;
		}
	}

	public function name() {
		return $this->name;
	}

	public function tableNames() {
		return $this->tableNames;
	}

	public function setTableNames($anArray) {
		$this->tableNames = $anArray;
	}
	/*
	 * Set the root entity type for an expanded entity 
	 * @param string containing the name of the entity type to be the root of an expanded entity
   	 */
	protected function setRootTypeName($anETypeName) {
		$this->rootTypeName = $anETypeName;
	}
	/*
	 * returns the name of the original entitytype, if this is an expanded type
	 * (see createExpandedType() )
	 */
	public function rootTypeName() {
		return $this->rootTypeName;
	}

	public function addAttributesFromTable() {
		global $db;
		global $errorStack;
		$res = $db->execSQL("SHOW COLUMNS FROM ".implode(",",$this->tableNames));
		if (!$db->OK()) {
			$errorStack->pushMsg(__file__.": Cannot fetch table attributes from DB.");
			// not necessary to display error here, done in open
			return;
		} else {
			while( $field = $db->fetchRow($res)) {
			        /*
			        Array
			        (
			            [Field] => id
			            [Type] => int(7)
			            [Null] =>
			            [Key] => PRI
			            [Default] =>
			            [Extra] => auto_increment
			        )
			        */
			        if ($field["Key"] == "PRI") {
			        	$addFunct = "addPrimaryKey";
			        } else {
			        	$addFunct = "add";
			        }
			        if ((substr($field["Type"],0,3) == "int") or
			        	(substr($field["Type"],0,3) == "dec") or
			        	(substr($field["Type"],0,3) == "flo") or
			        	(substr($field["Type"],0,3) == "dou") ) {
			        	$ftype = "GGFNumAttribute";
			        }  else {
			        	$ftype = "GGFTextAttribute";
			        }
			        $this->$addFunct(new $ftype($field["Field"]));
			}		
		}              
	}                      
	                
	public function add($anAttribute) {
		$this->attribute[$anAttribute->name()] = $anAttribute;
	}               
                        
	public function addPrimaryKey($anAttribute) {
		$this->add($anAttribute);
		array_push($this->primaryKey,$anAttribute->name());
	}               
                        
	public function primaryKeyFilter($row) {
		// returns a filter for direct access of the entity represented by $row
		$f = new GGFSqlFilter();
		$i = 0; 
		foreach ($this->primaryKey as $fieldName) {
			$pkAttribute = $this->getAttributeNamed($fieldName);
			$f->add(new GGFSqlFilterClause("AND",$fieldName,"=",$pkAttribute->asSQL($row)));
			$i++;
		}       
		return $f;
	}
	/**
	 * 
	 * Returns a filter to read related entities
	 * @param unknown_type $row
	 * @param unknown_type $anRTypeName
	 */               
    public function foreignKeyFilter($row,$anRTypeName) {
		global $ERModel;
		
		$rt = $ERModel->relationshipTypeNamed($anRTypeName);
		
		$f = new GGFERFilter($rt->relatedTypeFrom($this));
		$fk = $rt->foreignKeyFrom($this);
		$pk = $rt->relatedTypeKeyFrom($this);
		$rel = $rt->relatedTypeFrom($this);
		$rtName = $rel->name();
		for ($i = 0; $i < count($fk); $i++) {
			$fkaName = $fk[$i]; 
			$pkaName = $pk[$i]; 
			$fkAttribute = $this->getAttributeNamed($fkaName);
			$f->add(new GGFSqlFilterClause("AND",$rtName.".".$pkaName,"=",$fkAttribute->asSQL($row)));
		}       
		return $f;
	}               
                        
	public function attributeNames() {
		$res = array();
		foreach ($this->attribute as $att) {
			array_push($res, $att->name());
		}       
		return $res;
	}

	public function attributes() {
		return $this->attribute;
	}

	public function setAttributes($anArray) {
		$this->attribute = $anArray;
	}

	public function externalAttributeNames() {
		$en = array();
		foreach ($this->attribute as $att) {
			array_push($en, $att->externalName());
		}
		return $en;
	}  
	
	
	//GGF3.1.1
	public function setJoinfilter($f) {
		// not sure, if this accessor is really needed
		$this->joinfilter = $f;
	}
        
    public function joinfilter() {
		return $this->joinfilter;
	}
                        
	public function setDefaultAttributeNames($anArray) {
		// set the default subset of attribute names to be displayed in browser windows
		$this->defaultAttributeNames = $anArray;
	}               
	
	public function defaultAttributeNames() {
		// return the default subset of attribute names to be displayed in browser windows
		if (isset($this->defaultAttributeNames)) {
			return $this->defaultAttributeNames;
		} else {
			return $this->attributeNames();
		}
	}               
	
	public function primaryKey() {
		return $this->primaryKey;
	}               
	                
	public function primaryKeyValuesString($objectRow, $seperator = " ") {
		// creates a string of all the primary key values combined
		// this is a standard representation of SQL-quoted, HTML encoded, blank separated values
		$val = "";
		foreach ($this->primaryKey as $pkName) {
			//echo $pkName;
			$pkAtt = $this->getAttributeNamed($pkName);
			//print_r($pkAtt);
			if ($val == "") {
				$val = htmlspecialchars($pkAtt->asSQL($objectRow),ENT_QUOTES,'iso-8859-1');
			} else {		
				$val = $val.$seperator.
				htmlspecialchars($pkAtt->asSQL($objectRow),ENT_QUOTES,'iso-8859-1');
				//$pkAtt->asSQL($objectRow);
			}
			//echo "<br>".$val;
		}       
		return $val;
	}
	                
	public function attributeNamesString() {
		//returns a comma separated string with column names
		$a = $this->attributeNames();
		if (count($a) > 0 ) {
			$str = $a[0];
			for ($i=1; $i<(count($a)); $i++) {
				$str = $str.', '.$a[$i];
			}
		} else {
			$str = "";
		}       
		return $str;
	}               
                        
	public function isValid($aRow=array(), $checkReferentialIntegrity=FALSE) {
		// override this in derived classes, if needed
		foreach ($aRow as $key => $val) {
			if (!isset($this->attribute[$key])) {
				return FALSE;
			} 
			if (!$this->attribute[$key]->isValid($val)) {
				return FALSE;
			};
		}       
		if ($checkReferentialIntegrity) {
			// check referential integrity (foreign keys)
			$rTypes = $this->relationshipTypes();
			foreach ($rTypes as $rel) {
				if ($rel->minToCardinalityFrom($this) > 0) {
					if ($this->countRelatedEntitiesVia($row,$rel) == 0) {
						return FALSE;
					}
				}
			}
		}	
		return TRUE;
	}               
	                
	public function relationshipTypes() {
		global $ERModel;
		return $ERModel->relationshipTypesFrom($this);
	}               
	                
	public function countRelatedEntitiesVia($aRow, $anRTypeName) {
		global $ERModel;
		$rel = $ERModel->relationshipTypeNamed($anRTypeName);
		return $rel->countRelatedEntitiesFrom($this, $aRow);			
	}               
	                
	public function relatedEntitiesVia($aRow, $anRTypeName) {
		global $ERModel;
		$rel = $ERModel->relationshipTypeNamed($anRTypeName);
		return $rel->relatedEntitiesFrom($this, $aRow);			
	}               
	                
	public function getAttributeNamed($aString) {
		return $this->attribute[$aString];
	}

	/**
	 * returns a relationship type to itself
	 * 
	 * funny thing, but useful in createDelete for instance
	 */
	public function selfRelationshipType() {
		return new GGFRelationshipType("is recursively me",
				$this,$this->primaryKey(),
				array(0,1,0,1),
				$this,$this->primaryKey());
	}
				
	public function createDelete($row) {
		/* sample usage:
			$et = $this->ERModel->entityTypeNamed($this->ETypeName);
			$db->execSQL($et->createDelete($row),$et->tableNames()[0])
		*/
		$rel = $this->selfRelationshipType();
		$f = $rel->filterFrom($this,$row);
		$tn = $this->tableNames();
		return "DELETE FROM ".$tn[0]." ". $f->whereClause();
	}
	
	//GGF3.1.1
	/**
	 * This generates an Entitytype that contains has foreign keys replaced with values of referenced entities
	 *
	 * All foreign keys with an x:1 relationship are replaced by the values of the
	 * Attributes with default attribute names. This can be used to produce a list
	 * containing cleartext instead of codes.
	 *
	 * @param array containing the relationshiptypes that should be expanded. If null, then all to-one relationships are collected from the model
   	 */
	public function createExpandedType($expansionRTs=array()) {

		//echo "all rt:";print_r($this->relationshipTypes());
		// collect tablenames
		if (count($expansionRTs)==0) {
			$toOne = array(); // collect all to one relationship types from this
			foreach ($this->relationshipTypes() as $rt) {
				if ($rt->maxToCardinalityFrom($this) == 1) { //identifying relationhip
					array_push($toOne, $rt);
				}
			}
		} else {
			$toOne = $expansionRTs;
		}
		//echo "to one:";print_r($toOne);

		$newtn = $this->tableNames;
		foreach ($toOne as $rel) {
			$foreignType = $rel->relatedTypeFrom($this);
			$role = $rel->roleNameFor($foreignType); // take the name of the foreignkey columns concatenated as a role and alias name of the related entity

			array_push($newtn, $foreignType->tableNames[0].' '.$role);
		}
		//echo "newtn:";print_r($newtn);

		$ex = new GGFEntityType($this->name."_expanded", $newtn);
		$ex->setRootTypeName($this->name);
		$pk = $this->primaryKey();

		// clone structure
		$role = $this->name;
		foreach($this->attributeNames() as $an) {
			$att = $this->getAttributeNamed($an);
			$newAttClass = get_class($att);
			$newAtt = new $newAttClass($role.'.'.$an,$att->externalName());
			if (in_array ( $an, $pk)) { // take overa pk attribute 
				$ex->addPrimaryKey($newAtt);
			} else {
				$ex->add($newAtt);
			}
		}

		$joinfilter = new GGFSqlFilter();
		
		// now add related columns
		foreach ($toOne as $rel) {
			$foreignType = $rel->relatedTypeFrom($this);
			$role = $rel->roleNameFor($foreignType); // take the name of the foreignkey columns concatenated as a role and alias name of the related entity
			foreach ($foreignType->defaultAttributeNames() as $an) {
				$att = $foreignType->getAttributeNamed($an);
				$newAttClass = get_class($att);
				$newAtt = new $newAttClass($role.'.'.$an,$att->externalName());
				$ex->add($newAtt);
			}
			$joinfilter->add($rel->joinFilterTo($foreignType));
		}
		//print_r($joinfilter);
		$ex->setJoinfilter($joinfilter);
		return $ex;			

	}
	//GGF3.3
	public function createUpdate($row) {
		//create the update statement , with data from $row
		
		//GGF3.2.2
		$tn = $this->tableNames();
		$sql = "UPDATE ".$tn[0]." SET ";
		$pk = $this->primaryKey();

		
   		
		$atts = $this->attributeNames();
		$c = "";
		foreach ($atts as $an) {
			if (in_array($an,$pk) ) {
			} else {
				$att = $this->getAttributeNamed($an);
				//print_r($an);
				
					
				$sql = $sql.$c." ".$an."=".$att->asSQL($row);
				if ($c=="") { 
					$c = ", "; 	
				}
			}
		}
		
		$rel = $this->selfRelationshipType();
		$f = $rel->filterFrom($this,$row);
		$sql = $sql." ". $f->whereClause();
 
		return $sql;
	}


	//GGF3.3
	public function createInsert($row) {
		//create the insert statement , with data from $row
		
		$tn = $this->tableNames();
		$sql = "INSERT INTO ".$tn[0]." SET ";
		$pk = $this->primaryKey();
		
		$atts = $this->attributeNames();
		$c = "";
		foreach ($atts as $an) {
			if (in_array($an,$pk) ) {
			} else {
				$att = $this->getAttributeNamed($an);
				//print_r($an);
				
					
				$sql = $sql.$c." ".$an."=".$att->asSQL($row);
				if ($c=="") { 
					$c = ", "; 	
				}
			}
		}
		
 
		return $sql;
	}


	public function createSelect($cols,$filter,$sorter= 0, $page = 0, $pageSize = 0, $distinct=TRUE, $addPrimaryKey=TRUE) {
	
   		if($pageSize == 0) { // small list, no limit
   			$limit = "";
   		} else {
   			$limit = sprintf(" LIMIT %u, %u",
   			((1+($page - 1)*$pageSize)- 1),
			$pageSize);
   		}
   		
   		//GGF3.1.1
   		$f = "";
   		$sf = $filter->asSQL();
   		$jf = $this->joinfilter->asSQL(); // this ensures that tables are properly joined
		//print_r($this->joinfilter);	
		//print_r($jf);	
		//print_r($sf);	
		
   		if ($sf == "()" ) {
   			if ($jf != "()" ) {
		   		$f = ' WHERE ('.$jf.') ';
		   	} 
   		} else {
   			if ($jf != "()" ) {
   				$f = ' WHERE ('.$jf.') AND ('.$sf.')';
   			} else { 
   				$f = ' WHERE ('.$sf.') ';
   			}
   		}
   		//print_r($f);	
		
		// note MySQL will not return the fully qualified name as column name if it has no alias
		$alias_cols = array();
		$comma = "";
		if (count($cols) > 0) {
			foreach ($cols as $col) {
				array_push($alias_cols, $col.' AS "'.$col.'"');
			}
			if ($addPrimaryKey == TRUE) {
				$comma =",";
			}
		}
		
		//GGF 3.1.3
		if ($distinct) {
			$d = ' DISTINCT ';
		} else {
			$d = '';
		}
		if ($addPrimaryKey == TRUE) {
			//$pk = implode(",", $this->primaryKey()); // this ensures, that columns are not completely empty
			$elements = array();
			foreach ($this->primaryKey() as $key) {
				array_push($elements, $key.' AS "'.$key.'"');
			}
			$pk = implode(",", $elements);
   		} else {
			$pk = "";
		}
   		
		if ($sorter == 0) {
			$order="";
		} else {
			$order = $sorter->orderBy();
		}
	 	return sprintf('SELECT %s %s %s %s FROM %s %s %s %s',
	 		$d,
   			$pk, // this ensures, that columns are not completely empty
   			$comma,
   			implode(",",$alias_cols),
			implode(",",$this->tableNames),
			$f,
			$order,
			$limit
			);
	}
	
	public function defaultFilter() {
		// an empty defaultFilter is already created in ctor
		return $this->defaultFilter;
	}
	
	public function setDefaultFilter($filter) {
		$this->defaultFilter = $filter;
	}
	
	public function defaultSorter() {
		// an empty defaultSorter is already created in ctor
		return $this->defaultSorter;
	}
	
	public function setDefaultSorter($sorter) {
		$this->defaultSorter = $sorter;
	}
		               
}                       

                        
/**                     
 * class describing a relationship type of an ERmodel
 * note: instances do not hold data-values but a description 
 * of the properties of relationships
 *                      
 * @package GGF         
 * @version 3.0         
 * @since 3.0           
 * @author Gerald Zincke, Austria
 * @copyright 2005 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.html#top
 */                     
class GGFRelationshipType {
	protected $name;
	protected $from;
	protected $fromKey;
	protected $to;  
	protected $toKey;
	protected $card; // this is an array with the following indices
			// 0: fromMin 1: fromMax 2:toMin 3:toMax
			// cardinality min = 0 or 1
			// max = 1 or 999999999 (for n)
	                
	function __construct($name,$fromEntityType,$foreignKey,$cardinality,$toEntityType,$toKey=array()) {
		$this->name = $name;
		$this->from = $fromEntityType;
		$this->fromKey = $foreignKey;
		$this->to = $toEntityType;
		$this->card = $cardinality;
		$this->toKey = $toKey;
	}               
	public function name() {
		return $this->name;
	}               
	public function from() {
		return $this->from;
	}               
	public function to() {
		return $this->to;
	}
	
	public function relatedTypeFrom($fromType) {
		if ($fromType == $this->from) { // normal direction
			return $this->to;
		} else {
			//reverse direction
			return $this->from;
		}
	} 
	public function foreignKeyFrom($fromType) {
		if ($fromType == $this->from) { // normal direction
			return $this->fromKey;
		} else {
			//reverse direction
			return $this->toKey;
		}
	}
	public function relatedTypeKeyFrom($fromType) {
		if ($fromType == $this->from) { // normal direction
			return $this->toKey;
		} else {
			//reverse direction
			return $this->fromKey;
		}
	}	               
	public function minToCardinalityFrom($fromType) {
		if ($fromType == $this->from) { // normal direction
			return $this->card[2];
		} else {
			return $this->card[0];
		}       
	}	        
        public function maxToCardinalityFrom($fromType) {
		if ($fromType == $this->from) { // normal direction
			return $this->card[3];
		} else {
			return $this->card[1];
		}       
	}	        
                        
	public function countRelatedEntitiesFrom($anEType, $aRow) {
		global $db;
		global $errorStack;
		$f = $this->filterFrom($anEType,$aRow);
		if ($anEType == $this->from) { // normal direction
			$to = $this->to;
		} else {
			//reverse direction
			$to = $this->from;
		}       
		$sql = "SELECT COUNT(*) FROM ".$to->table().$f->whereClause();
		$res = $db->execSQL($sql);
		if (!$db->OK()) {
			$errorStack->pushMsg(__File__.": Cannot count related entities .");
			// not necessary to display error here, done in open
			return 0;
		} else {
			$row = $db->fetchRow($res);
			return $row["count(*)"];
		}       
	}               
	                
	public function relatedEntitiesFrom($anEType, $aRow) {
		global $db;
		global $errorStack;
		
		$f = $this->filterFrom($anEType,$aRow);
		if ($anEType == $this->from) { // normal direction
			$to = $this->to;
		} else {
			//reverse direction
			$to = $this->from;
		}       
		$sql = "SELECT ".$to->attributeNamesString()." FROM ".implode(",",$to->tableNames())." ".$f->whereClause();
		$res = $db->execSQL($sql);
		if (!$db->OK()) {
			$errorStack->pushMsg(__File__.": Cannot count related entities .");
			// not necessary to display error here, done in open
			return array();
		} else {
			$result = array();
			while($row = $db->fetchRow($res)) {
				array_push($result, $row);
			}
			return $result;
		}
	}               
	      
	public function roleNameFor($anEType) {
		if ($anEType == $this->from) { // normal direction
			$tk = $this->fromKey;
		} else {
			//reverse direction
			$tk = $this->toKey;
		}       
	        return implode("_",$tk);
	}
	                
   	/**
   	 * creates a filter used to join the two entitytypes
	 *
	 * creates comparison of the form foreignkey1 = alias.key1 AND foreignkey2 = alias.key2 
   	 *
   	 * @param GGFEntitytype containing the entity type, that is target of the relationship   	 
   	 */
	public function joinFilterTo($foreignType) {
		if ($foreignType == $this->to) { // normal direction
			$fk = $this->toKey;
			$to = $this->from;
			$tk = $this->fromKey;
		} else {
			//reverse direction
			$fk = $this->fromKey;
			$to = $this->to;
			$tk = $this->toKey;
		}       
		$toRole = $this->roleNameFor($foreignType);
		$f = new GGFSqlFilter();
		$i = 0; 
		//print_r($fk);
		
		foreach ($tk as $toFieldName) {
			//$toAttribute = $to->getAttributeNamed($toFieldName);
			$tempArray = array();
			$tempArray[$toFieldName] = $fromValue[$fk[$i]];
	
			$f->add(new GGFSqlFilterClause("AND",$this->to->name().'.'.$toFieldName,"=",$toRole.'.'.$fk[$i],FALSE));			

			$i++;
		}       
		//print_r($f);   		        
   		        
		return $f;
	}               

	public function filterFrom($fromType,$fromValue) {
		if ($fromType == $this->from) { // normal direction
			$fk = $this->fromKey;
			$to = $this->to;
			$tk = $this->toKey;
		} else {
			//reverse direction
			$fk = $this->toKey;
			$to = $this->from;
			$tk = $this->fromKey;
		}       
		$f = new GGFERFilter($to);
		$i = 0; 
		foreach ($tk as $toFieldName) {
			$toAttribute = $to->getAttributeNamed($toFieldName);
			$tempArray = array();
			$tempArray[$toFieldName] = $fromValue[$fk[$i]];
			$f->add(new GGFSqlFilterClause("AND",$toFieldName,"=",$toAttribute->asSQL($tempArray)));
			$i++;
		}       
		return $f;
	}               
	                
}                       
                        
/**                     
 * class describing the apecial relationship between objects and notes
 * note: This is not a simple foreign-key relationship, because the attribute
 * objectkey in the GGFNote table holds a text-composition of the primary key of the object. 
 *                      
 * @package GGF         
 * @version 3.0         
 * @since 3.0           
 * @author Gerald Zincke, Austria
 * @copyright 2005 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.html#top
 */                     
class GGFObjectNotesRelationshipType extends GGFRelationshipType {
	// the relationship between objects and their notes is not a simple =
                        
	public function filterFrom($objectType,$objectRow) {
		$f = new GGFSqlFilter();
		if ($objectType == $this->from) { // normal direction
			$fk = $this->fromKey;
			$val = $objectType->primaryKeyValuesString($row);
			$f->add(new GGFSqlFilterClause("AND",$fk,"=","'".$val."'"));
			return $f;
		} else {
			//reverse direction 
			// parse the objectkey in the note
			$val = $objectRow[$this->toKey[0]];
			
			$keyvalenc = explode (" ", $val);
			$keyval = array();
			foreach ($keyvalenc as $str) {
				array_push($keyval,html_entity_decode( $str,ENT_QUOTES,'iso-8859-1'));
			}
			$i = 0;
			foreach ($this->fromKey as $pkName) {
				$f->add(new GGFSqlFilterClause("AND",$pkName,"=","'".$keyval[$i]."'"));
				$i++;
			}
			return $f;
		}       
	}               
	                
}                       
                        
class GGFERModel {      
	// any window cann access the ER model via
	// $this->ERModel
	//
	// entitytypes can be accessed via: $this->ERModel->entityTypeNamed('aname')
	// relationship types are found via:$this->ERModel->relationshipTypeNamed('aname')
	//
	//create one derived class per application
	//instantiate a singleton $model
	//
	// Note: The ERModel is bound to the session. Changes in your "initialize" method
	// will have no effect until the app is restarted
	//
	//the $ERmodel is global
	public $entityType = array();
	protected $relationshipType = array(); 
                        
	function __construct() {
		global $ERModel;
		$ERModel =& $this;
		$this->initialize();
	}               
	                
	protected function initialize() {
		global $ERModel;
		$ERModel = $this;
		// in derived classes create the model here
		// see GGFDemoModel.php
		/*      
		
		note: in future we can also use the function 
		GGFEntityType->addAttributesFromTable
		to add the attributes directly from the database table
		
		CREATE TABLE GGFNote
		(	id		integer AUTO_INCREMENT,
		last_change 	TIMESTAMP,
		userid  	VARCHAR(16) NOT NULL,
		ispublic        BOOL,
		objectkey       VARCHAR(64),
		windowclassname VARCHAR(64) NOT NULL,
		controlname	VARCHAR(64),
		note		TEXT,
		PRIMARY KEY (id)
		)               
		*/      
		$note = new GGFEntityType("GGFNote");
		$note->addPrimaryKey(new GGFNumAttribute("id"));
		$note->add(new GGFTextAttribute("last_change"));
		$note->add(new GGFTextAttribute("userid"));
		$note->add(new GGFNumAttribute("ispublic"));
		$note->add(new GGFTextAttribute("objectkey"));
		// note the object key is a string that holds an array of primarykey values of an object
		// the values are SQL quoted and html encoded (with htmlentities($pkAtt->asSQL($objectRow),ENT_QUOTES,'iso-8859-1')
		// and separated by blanks (see class GGFObjectNotesRelationshipType, function filterFrom)
		$note->add(new GGFTextAttribute("windowclassname"));
		$note->add(new GGFTextAttribute("controlname"));
		$note->add(new GGFTextAttribute("note"));
		$this->add($note);
		
		$desc = new GGFEntityType("GGFQuery",array("GGFQuery"));
		$desc->addPrimaryKey(new GGFNumAttribute("id")); /* int */
		$desc->add(new GGFTextAttribute("owner")); /* varchar */
		$desc->add(new GGFTextAttribute("ETypeName")); /* varchar */
		$desc->add(new GGFTextAttribute("browser")); /* varchar */
		$desc->add(new GGFTextAttribute("qname")); /* varchar */
		$desc->add(new GGFTextAttribute("filter")); /* text */
		$desc->add(new GGFTextAttribute("sorter")); /* text */
		$desc->add(new GGFTextAttribute("thecolumns")); /* text */
		$desc->add(new GGFTextAttribute("columnsize")); /* text */
		$desc->add(new GGFTextAttribute("thechecksum")); /* text */
		
		$desc->setDefaultAttributeNames(array(id, owner, ETypeName, qname, filter, sorter, columns)); 
		$this->add($desc);
		
	}               
	                
	public function add($et) {
		$this->entityType[$et->name()] = $et;
	}               
                        
	protected function addRel($rt) {
		$this->relationshipType[$rt->name()] = $rt;
	}               
                        
	public function entityTypeNamed($anETypeName) {
		global $errorStack;
		foreach ($this->entityType as $name => $ent) {
			//echo $name;
			if ($anETypeName == $name ) {
				return $ent;
			}
		}       
		$errorStack->pushMsg(__File__.": Cannot find entity type '".$anETypeName."'");
		//debug_print_backtrace();
	}               
                        
	public function relationshipTypeNamed($anRTypeName) {
		global  $errorStack;
		foreach ($this->relationshipType as $name => $rel) {
			if ($anRTypeName == $name ) {
				return $rel;
			}
		}       
		$errorStack->pushMsg(__File__.": Cannot count find relationship type ".$anRTypeName);
	}
	
	public function relationshipTypesFrom($et) {
		
		$res = array();
		foreach ($this->relationshipType as $rel) {

			if ($rel->to() == $et) {
				array_push($res,$rel);
			} elseif ($rel->from() == $et) {
				array_push($res,$rel);
			}
		}   
		return $res;    
	}               
	                       
}                              
                               
                               
?>