<?PHP

/**
 * GGFControl base class for all single user intzerface controls.
 * A control is an object that manages output (rendering to HTML) and 
 * input (processing client requests) of fields and other elements on a window or dialog. 
 * Controls are added to ControlPanes. ControlPanes can be added to 
 * other ControlPanes . Each window contains one master pane.
 * 
 * @todo 
 * @abstract
 * @package GGF
 * @version 4.2
 * @since 2.0
 * @author Gerald Zincke, Austria
 * @copyright 2005-2011 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/GGF.htm#License
 */
class GGFControl {
	// this models an active user interface element, abstract base class
	// controls are managed by GGControlContainers
	// a container can render its controls (returning an HTML string)
	// active controls (buttons) know their callback and their validation function	

	/** @var string the HTML generated */
	protected $html;
	
	/** @var string the name of the control */
	protected $name;
	
	/** @var mixed the value displayed or read */
	protected $value;
	
	/** @var integer the horizontal size of the control */
	protected $size;
	
	/** @var string style etc */
	protected $extraHTML;
	
	/** @var string tip text to be displayed in the ballon help */
	protected $tiptext="";
	
	/** @var GGFControlPane reference to the pane where the control is stored*/
	protected $myContainer;
	
	/** @var GGFControlWindow the window where the control is displayed */
	protected $myWindow;
	
	/** @var integer the tabulator index of the control, for future use */
	protected $tabindex;
	
	/** @var boolean tells if the control is disabled */
	protected $isDisabled=FALSE;

	/** @var boolean tells if the control is read only */
	protected $readonly=FALSE;

	/** @var boolean tells if the control is mandatory */
	protected $mandatory=FALSE;

	/** @var string RRGGBB string for foreground color. Empty string: default */
	protected $Foreground="";
	
	/** @var string RRGGBB string for background color. Empty string: default */
	public  $Background="";
	
	/** @var string font family specification. default: same as parent */
	protected $fontFamily="";

	/** @var string font size specification including unit. default: same as parent */
	protected $fontSize="";
	
	/** @var array font style options set. default: same as parent */
	protected $fontStyle=array(); 
	
	/** @var array font style options set. default: same as parent */
	protected $lineHeight="";
	
	
	/**
   	 * abstract constructor of base class of all controls
   	 *
   	 * A control is an object that encapsulates HTML that forms a basic
   	 * GUI element, for instance a listbox.
   	 *
   	 * @param string containing the name of the control, used to retrieve it from a pane
      	 * @param string value, depends on the nature of the control, what it is. Is a string for an entry field
      	 * @param integer optionsl horizontal size, depends on the nature of the control, what scale it is. Is number of chars for an entry field
      	 * @param string optional tip text (balloon help)
      	 * @param string optional HTML for instance to modify the style of the control
      	 * 
   	 */
	function __construct(
		$name, 
		$value, 
		$size=0, 
		$tiptext="", 
		$extraHTML="",
		$mandatory=false) {
	
		$this->setHTML();
		$this->name = $name;
		$this->value = $value;
		$this->size = $size;
		$this->tiptext = $tiptext;
		$this->extraHTML = $extraHTML;
		$this->mandatory = $mandatory;
	}
	/**
   	 * @return HDDControlDescriptor an object that describes the control 
   	*/
	public function createDescriptor() {
		$cd = new HDDControlDescriptor(get_class($this),$this->name);
		$props = $cd->properties();
		foreach ($props as $pn => $pv) {
			$props[$pn] = $this->$pn;
		}
		$cd->setProperties($props);
		return $cd;
	}
	
	/**
   	 * returns a string that can be used to instantiate the control 
   	*/
	public function constructorString() {
		return 
		'"'.$this->name.'"'.
		'"'.$this->value.'"'.
		'"'.$this->size.'"'.
		'"'.$this->tiptext.'"'.
		'"'.$this->extraHTML.'"'.
		'"'.$this->mandatory.'"';
	}
	
	

	public function valueType(){
		return "string";
	}
	/**
	 * 
	 * @deprecated
	 * 
	 */
	public function initializeProperties($pValues){
		$v = array();
		$r = array();
		foreach ($pValues as $key => $value) {
			if ($value == "") {
				$r[] = $key;
			} else {
				$v[$key] = $value;
			}
		}
		$this->resetP($r);
		$this->initP($v);
	}
	/**
	 * 
	 * resets property values of the control
	 * @param array $props array with instance variable names
	 */
	public function resetP($props){
		foreach ($props as $pName) {
			if (!(in_array($pName, array('html', 'html2', 'myWindow')))) {
				if ($pName == "value") {
					if ($this->valueType() == "array") {
						$this->value = array();
					} else {
						$this->value = "";
					}
				} elseif ($pName == "selections") {
					if ($this->selectionsType() == "array") {
						$this->selections = array();
					} else {
						$this->selections = "";
					}
				} elseif ($pName == "keys") {
					if ($this->valueType() == "array") {
						$this->keys = array();
					} else {
						$this->keys = "";
					}
				} else {
					$this->$pName = "";
				}
			}
		}
	}

	/**
	 * 
	 * initialize property values of the control
	 * @param array $pValues array with instance variable names => values
	 */
	public function initP($pValues){
		foreach ($pValues as $pName => $pValue) {
			if (!(in_array($pName, array('html', 'html2', 'myWindow')))) {
				if ($pName == "value") {
					if ($this->valueType() == "array") {
						$this->value = explode("|", $pValue);
					} else {
						$this->value = $pValue;
					}
				} elseif ($pName == "selections") {
					if ($this->selectionsType() == "array") {
						$this->selections = explode("|", $pValue);
					} else {
						$this->selections = $pValue;
//						print_r($this);
					}
				} elseif ($pName == "keys") {
					if ($this->valueType() == "array") {
						$this->keys = explode("|", $pValue);
					} else {
						$this->keys = $pValue;
					}
				} else {
					$this->$pName = $pValue;
				}
			}
		}
	}
	
	public function isFrame() {
		return FALSE;
	}
	
	protected function setHTML() {
		$this->html = '';
	}

	public function setReadonly($bool=TRUE) {
		// note: this is function-less for many controls
		$this->readonly = $bool;
	}
	public function isReadonly() {
		return $this->readonly;
	}

	public function setMandatory($bool=TRUE) {
		// note: this is function-less for many controls
		$this->mandatory = $bool;
	}
	public function isMandatory() {
		return $this->mandatory;
	}

	public function isPane() {
		return FALSE;
	}

	public function isButton() { // tells if its a form submit button
		return FALSE;
	}	
	
	public function isEntry() {
		return FALSE;
	}
	public function isStyleable() {
		return TRUE;
	}
	//GGF 3.2
	public function setExtraHTML($html) {
		$this->extraHTML = $html;
	}
	
	public function setDisabled($on = TRUE) {
		$this->disabled = $on;
	}
	
	public function setTabindex($ind) {
		$this->tabindex = $ind;
	}
	
	public function setContainer($aContainer) {
		$this->myContainer = $aContainer;
	}
	
	public function setAutofocus() {
	}
	
	public function container() {
		return $this->myContainer;
	}
	
	public function setWindow($aWindow) {
		$this->myWindow = $aWindow;
	}
	
	public function name() {
		return $this->name;
	}
	
	public function value() {
		return $this->value;
	}
	
	public function setValue($aValue) {
		$this->value = $aValue;
	}
	
	public function tipText() {
		return $this->tiptext;
	}
	
	public function setTipText($aTipText) {
		$this->tiptext = $aTipText;
	}
	
	public function setForeground($aString) {
		$this->Foreground = $aString;
	}
	
	public function setBackground($aString) {
		$this->Background = $aString;
	}
	
	
	protected function controlStyle() {
		$style=" ";
		if ($this->Background > "") {
			$style=$style."background-color:".$this->Background.";";
		}
		if ($this->Foreground > "") {
			$style=$style."color:".$this->Foreground.";";
		}
		
		if ($this->fontFamily > "") {
			$style = $style.' font-family:'.$this->fontFamily."; ";
		}

		if (isset($this->fontStyle["bold"])) {
			if ($this->fontStyle["bold"]) {
				$style = $style.' font-weight:bold; ';
			}
		}
		if (isset($this->fontStyle["italic"])) {
			if ($this->fontStyle["italic"]) {
				$style = $style.' font-style:italic; ';
			}
		}
		if (isset($this->fontStyle["underline"])) {
			if ($this->fontStyle["underline"]) {
				$style = $style.' text-decoration:underline; ';
			}
		}	
		
		if ($this->lineHeight > "") {
			$style = $style.' line-height:'.$this->lineHeight."; ";
		}
		if ($this->fontSize > "") {
			$style = $style.' font-size:'.$this->fontSize."; ";
		}
		
		return $style;
	}
	
	public function processRequest() {
		// nothing to do over write this for input controls that read a value from screen
	}
	protected function controlID() {
		// this should be unique per window. t.b.d.
	}
	public function render($demo=FALSE,$index=1,$indent=0) {
		$prefix = str_pad("",1+$indent*3);
		return $prefix.sprintf($this->html, $this->name, $this->value, $this->size, $this->tiptext, $this->extraHTML);
	}
	public function editRender() {
		// this is  a preliminary implementation
		return '<fieldset><legend><b>'.$this->name.'</b></legend>
		<table><tr>
			<td><table>
			   <tr><td><a href="'.$this->myWindow->callbackURL("_eventEditControl").'&_controlID='.$this->controlID().'">e</a><td></tr>
			   <tr><td><a href="'.$this->myWindow->callbackURL("_eventDeleteControl").'&_controlID='.$this->controlID().'">d</a><td></tr>
			</table></td>
			<td>'.$this->render().'</td>
		</tr></table></fieldset>';
	}
	
	//GGF3.6
	public function isSingleSelection() {
		return FALSE;
	}

	public function isForWindow() {
		// tells, if the control may be used in a window
		return TRUE;
	}

	protected function prettyString($string, $width) {
		$ret = substr($string, 0, $width);
		return $ret.str_repeat("&nbsp;", ($width - strlen($ret)));
	}

	protected function prettyColumn($fields, $colwidths, $delimiter) {
		$ret = $this->prettyString($fields[0], $colwidths[0]);
		for ($i = 1; $i < count($fields); $i++) {
			$ret = $ret.$delimiter.$this->prettyString($fields[$i], $colwidths[$i]);
		}
		return $ret;
	}
}

/**
 * May be used to prepare an arbitrary HTML construct and encapsulate it in
 * a GGFControl object. This allows to avoid writing HTML outside Control classes.
 * 
 * @todo version, test
 * @package GGF
 * @version 3.2
 * @since 2.3
 * @author Gerald Zincke, Austria
 * @copyright 2005 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.htm#top
 */
class GGFGenericControl extends GGFControl {
	
//	protected $html;
//	protected $name;
//	protected $value;
//	protected $size;
//	protected $extraHTML;
//	protected $tiptext="";
//	protected $myContainer;
//	protected $tabindex;
//	protected $isDisabled=FALSE;
	protected $input;
	protected $requestProcessor;
	protected $renderer;

	/**
   	 * constructor for generic control
   	 *
   	 * A generic control is an object that encapsulates HTML that is
   	 * provided in its constructor. Use this only if no other control
   	 * class fits.
   	 *
   	 * @param string containing the name of the control, used to retrieve it from a pane
     * @param string HTML string that is used to display the control
     * @param string optional name of a window function that is able to process the request data (input from the control)
     * @param string optional name of a window function that will return the HTML string for the control; should observe: isDisabled, tabindex, tiptext 
     * @param string optional tip text (balloon help)
     * @param bool tells if this is an input control
     * 
   	 */
	function __construct($name, $html, $requestProcessor="", $renderer="", $tiptext="", $input=FALSE) {
		// requestProcessor: name of a global function that will transfer $_REQUEST data into the control
		// renderer: name of a global function that will return the HTML string for the control
		//           should observe: isDisabled, tabindex, tiptext
		$this->html = $html;
		$this->requestProcessor = $requestProcessor;
		$this->name = $name;
		$this->tiptext = $tiptext;
		$this->input = $input;
		$this->renderer = $renderer;
		
	}
	public function isEntry() {
		return $this->input;
	}
		
	public function isStyleable() {
		return FALSE;
	}
	
	
	public function processRequest($mc) {
		if (!$this->requestProcessor=="") {
			$this->requestProcessor($this,$mc);
		}
	}
	public function render($demo=FALSE,$index=1,$indent=0) {
		$prefix = str_pad("",1+$indent*3);
		
		/* if ($demo) { // do not do this now, because this class is used for p, br, enpane etc.
			return "&lt;Demo Placeholder:".$this->name."&gt;";
		} else { */
		if (!$this->renderer=="") {
			return $prefix.$this->renderer->render($demo,$index,$indent);
		}
		return $prefix.$this->html;

	}
	public function editRender() { // not supported
		return $this->render();
	}
	//GGF 3.2
	public function setValue($html) {
		$this->html = $html;
	}
}

/**
 * models a text area in a window that is not about to be changed by the user
 * 
 * @todo version, test
 * @package GGF
 * @version 2.3
 * @since 2.3
 * @author Gerald Zincke, Austria
 * @copyright 2005 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.htm#top
 */
class  GGFStaticfield extends GGFControl {

//	protected $html;
//	protected $name;
//	protected $value;
//	protected $size;
//	protected $extraHTML;
	
	function __construct($name, $value, $size=0, $tiptext="", $extraHTML="") {
		//name may also be "", tiptext is here for future use
		parent::__construct($name, $value, $size, $tiptext, $extraHTML);
	}
		
	protected function setHTML() {
		$this->html = '<span  class="'.get_class($this).'" id="%s" style="%s">%s</span>';
	}

	public function processRequest($myContext) {
		// nothing to do
	}

	public function render($demo=FALSE,$index=1,$indent=0) {
		$prefix = str_pad("",1+$indent*3);
		
		$extra = $this->controlStyle();
		if ($this->size > 0) {
			$extra = $extra.sprintf(' width:%dpx; overflow:hidden ', $this->size );
		} 
		$extra = $extra.$this->extraHTML;
		return $prefix.sprintf($this->html, $this->name, $extra, htmlspecialchars($this->value));
	}
}

/**
 * models a normal hyperlink
 * 
 * @todo version, test
 * @package GGF
 * @version 3.1
 * @since 2.3
 * @author Gerald Zincke, Austria
 * @copyright 2005 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.htm#top
 */
class  GGFStaticHyperlinkfield extends GGFStaticfield {
	// this models a hyperlink that cannot be modified by the user

//	protected $html;
//	protected $name;
//	protected $value;
//	protected $size;
//	protected $extraHTML;
	protected $url;
	protected $target="_blank";
	
	function __construct($name, $url, $text="", $size=0, $tiptext="", $extraHTML="", $target="") {
		//name my also be "", tiptext is here for future use
		if ($text == "") {
			parent::__construct($name, $value=$url, $size, $tiptext, $extraHTML);
		} else {
			parent::__construct($name, $value=$text, $size, $tiptext, $extraHTML);
		}
		$this->url = $url;
		$this->target = $target;
	}
		
	protected function setHTML() {
		$this->html = '<a  class="'.get_class($this).'" id="%s" href="%s"  title="%s"  target="%s" style="%s">%s</a>';
	}
	//GGF3.1
	public function setURL($aURL) {
		$this->url = $aURL;
	}

	public function render($demo=FALSE,$index=1,$indent=0) {
		$prefix = str_pad("",1+$indent*3);

		if ($this->target=="") {
			$target="_blank";
		}
		$extra = $this->controlStyle();
		if ($this->size > 0) {
			$extra = $extra.sprintf(' width:%dpx; overflow:hidden ', $this->size );
		} 
		$extra = $extra.$this->extraHTML;
		
		return $prefix.sprintf($this->html, $this->name, $this->url, $this->tiptext, $target, $extra, htmlspecialchars($this->value));
	}
}

class  GGFImagefield extends GGFStaticHyperlinkfield {

	protected $vsize;
	
	function __construct($name, $url, $text="", $hsize=0, $vsize=0, $tiptext="", $extraHTML="") {
		parent::__construct($name, $url, $text, $hsize, $tiptext, $extraHTML);
		$this->vsize =$vsize;
	}
	
	protected function setHTML() {
		$this->html = '<img  class="'.get_class($this).'" id="%s" src="%s"  title="%s" alt="%s" style="%s">';
	}
	public function render($demo=FALSE,$index=1,$indent=0) {
		$prefix = str_pad("",1+$indent*3);

		$extra = ' ';
		if ($this->size > 0) {
			$extra = $extra.sprintf(' width:%s; ', $this->size );
		} 
		if ($this->vsize > 0) {
			$extra = $extra.sprintf(' height:%s; ', $this->vsize );
		} 
		$extra = $extra.$this->extraHTML;
		
		return $prefix.sprintf($this->html, $this->name, $this->url, $this->tiptext, htmlspecialchars($this->value),$extra);
	}
}	
class  GGFImageHyperlinkfield extends GGFStaticHyperlinkfield {

	protected $vsize;
	protected $imageUrl;
	protected $callback; // @todo check if needed
	
	
	function __construct($name, $url, $imageUrl, $hsize=0, $vsize=0, $tiptext="", $extraHTML="", $target="") {
		parent::__construct($name, $url, $text="", $hsize, $tiptext, $extraHTML, $target);
		$this->vsize =$vsize;
		$this->imageUrl = $imageUrl;
	}
	
	public function callback() {
		return $this->callback;
	}
	
	public function setCallback($amethod) {
		$this->callback = $amethod;
	}
	
	protected function setHTML() {

		$this->html = '<a class="'.get_class($this).'" id="%s" href="%s"  title="%s" target="%s" style="%s"><img src="%s"  alt="%s" %s></a>';
	}

	public function render($demo=FALSE,$index=1,$indent=0) {
		$prefix = str_pad("",1+$indent*3);
		$win = $this->myWindow;
		
		$url = $this->url;
		if ($demo) {
			$url = $win->callbackURL("eventDemoClicked");
			$this->target = "_blank";
		} else {
			if (GGFControlWindow::$staticCallbacks) {
				if ($this->url == "") {
					$url = $win->callbackURL($this->callback);
				}
			} else {
				$url = $win->callbackURL($this->callback);
			}
		}
		

		$extra = ' ';
		if ($this->size > 0) {
			$extra = $extra.sprintf(' width="%d" ', $this->size );
		} 
		if ($this->vsize > 0) {
			$extra = $extra.sprintf(' height="%d" ', $this->vsize );
		} 
		$extra = $extra.$this->extraHTML;
		
		return $prefix.sprintf($this->html, $this->name, $url, $this->tiptext, $this->target, $this->extraHTML , $this->imageUrl, htmlspecialchars($this->value),$extra);
	}
}	
/**
 * models a pseudo pushbutton for opening new windows (target=_blank) or for
 * use in a window, where normal pushbuttons cannot be used.
 * derived from a hyperlink field, but knows pushbutton protocol
 * 
 * @todo remove setHTML; replace get_class in parent by an instance-specific method
 * @package GGF
 * @version 3.6
 * @since 3.6
 * @author Gerald Zincke, Austria
 * @copyright 2007 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.htm#top
 */
class  GGFPseudoPushbutton extends GGFStaticHyperlinkfield {

	protected $callback; // @todo check if needed
	protected $validator;
	protected $size;

	function __construct($name, $buttontext, $callback, $validator="", $size=0, $tiptext="", $extraHTML="", $target="_blank") {
		parent::__construct($name, "no url yet",$buttontext, 0, 
					$tiptext, 
					$extraHTML);
		$this->callback = $callback;
		$this->validator = $validator;
		$this->size = $size;
		$this->target = $target;
	}	
		
	public function callback() {
		// note the url instance variable does not store the complete url, only the name of the
		// callback function. The complete URL is composed on rendering time
		return $this->callback;
	}
	
	public function setCallback($amethod) {
		$this->callback = $amethod;
	}
	
	public function isButton() { // tells if its a form submit button
		return FALSE;        // we are only a pseudo button, so no form data is processed
	}	

	public function value() {
		return $this->value;
	}

	public function setTarget($str) {
		$this->target=$str;
	}
	
	protected function setHTML() {
		$this->html = '<a  class="'.get_class($this).'"  href="%s"  title="%s" target="%s" style="%s">%s</a>';
	}
	
	public function render($demo=FALSE,$index=1,$indent=0) {
		$prefix = str_pad("",1+$indent*3);
		
		$win = $this->myWindow;
		$val = $this->validator;
		$target = $this->target;
		
		
		if ($demo) {
			$this->url = $win->callbackURL("eventDemoClicked");
			$target = "_top";
		} else {
			if (GGFControlWindow::$staticCallbacks) {
				if ($this->url == "") {
					$this->url = $win->callbackURL($this->callback);
				}
			} else {
				$this->url = $win->callbackURL($this->callback);
			}
		}
		
		$extra = $this->controlStyle();
		
		if ($this->size > 0) {
			$extra = $extra.sprintf(' width:%dpx; overflow:hidden ', $this->size).$this->extraHTML;
		} else {
			$extra = $extra.$this->extraHTML;
		}	
		
		if (($val=="") || (GGFControlWindow::$staticCallbacks)) {
			return $prefix.sprintf($this->html,  $this->url, $this->tiptext, $target, $extra, htmlspecialchars($this->value));
		} else {
			if (!$val =="") {
				if ($win->$val()) { //$validator must be the name of a function returning true when the button should be active
					return $prefix.sprintf($this->html, $this->url, $this->tiptext, $target, $extra, htmlspecialchars($this->value));
				} else {
					// make placeholder
					//return $this->value;
					return $prefix.sprintf('<div  style="%s">%s</div>', $extra, htmlspecialchars($this->value));
				}
			}
		}
	}
}

/**
 * models a text area containg rows. each of them a hyperlink to
 * update the row. Each line also contains a hyperlink to delete it.
 * May also have a hyperlink to copy a row
 * The headline may also contain a hyperlink to sort
 * 
 * @todo 
 * @package GGF
 * @version 3.2.2
 * @since 2.3
 * @author Gerald Zincke, Austria
 * @copyright 2005,2006 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.htm#top
 */
class GGFListArea extends GGFStaticfield {
	// can display an array of database rows 
	// value must be an array
	// provide the names of the columns by setColumns()
	
	protected $columns;
	protected $columnsize=array();
	protected $externalColumnNames;
	protected $deleteCallback;
	protected $updateCallback;
	protected $updateTarget;
	protected $viewCallback;
	protected $viewTarget;
	
	protected $copyCallback;
	protected $sortCallback;
	
	/**
   	 * creates a control for scrolling through database tables
   	 *
   	 * @param string name of control
   	 * @param array names of the columns
   	 * @param array the database rows to display 
   	 * @param string name of the window-function for handling the update-event 
   	 * @param string name of the window-function for handling the delete-event 
   	 * @param string name of the window-function for handling the copy-event (may also be empty)
	 * @param string name of the window-function for handling the sort-event (may also be empty) 
	 * @param integer number of rows in a page 
	 * @param string tiptext describing the control 
	 * @param string optional html code 
	 */
	function __construct($name="aGGFListPane", $columns=array(), $rows=array(), 
		$updateCallback="eventUpdate", $deleteCallback="eventDelete", 
		$copyCallback="", $sortCallback="", $pageSize=0, $tiptext="", $extraHTML="") {
		//name my also be "", tiptext is here for future use
		parent::__construct($name, $rows, $pageSize, $tiptext, $extraHTML);
		$this->columns = $columns;
		$this->updateCallback=$updateCallback;
		$this->deleteCallback=$deleteCallback;
		$this->copyCallback=$copyCallback;
		$this->sortCallback=$sortCallback;

		$this->updateTarget = '_top';
		
		// $value = $rows;
	}

	public function isStyleable() {
		return FALSE;
	}

	public function setColumns($columns) {
		$this->columns = $columns;
	}

	/**
	  * set special columnsizes for specific columns
	  *
	  * @param array indexed by columnname, gives size of column	 
	  */
	public function setColumnSize($columnsize) {
		$this->columnsize = $columnsize;
	}

	/*
	 * This defines an additional callback upon records displayed
	 */
	public function setViewCallback($function, $target="_blank") {
		$this->viewCallback = $function;
		$this->viewTarget = $target;
	}

	public function setUpdateCallback($function, $target="_top") {
		$this->updateCallback = $function;
		$this->updateTarget = $target;
	}

	public function externalColumnNames() {
		if (!isset($this->externalColumnNames)) {
			return $this->columns;
		} 
		return $this->externalColumnNames;
	}

	public function setExternalColumnNames($columns) {
		$this->externalColumnNames = $columns;
	}

	/** 
	 * render the control to HTML
	 *
	 * Thanks to Simon Opelt, the performance has been increased
	 */
	public function render($demo=FALSE,$index=1,$indent=0) {
		$prefix = str_pad("",1+$indent*3);
		
		global $maxColumnSize;

		if ($demo) {
			$sortpath = $this->myWindow->callbackURL("eventDemoClicked");
			$updatepath = $sortpath;
			$viewpath = $sortpath;
			$copypath = $sortpath;
			$deletepath = $sortpath;
		} else {
			$sortpath = $this->myWindow->callbackURL($this->sortCallback);
			$updatepath = $this->myWindow->callbackURL($this->updateCallback);
			$viewpath = $this->myWindow->callbackURL($this->viewCallback);
			$copypath = $this->myWindow->callbackURL($this->copyCallback);			
			$deletepath = $this->myWindow->callbackURL($this->deleteCallback);			
		}
		$win = $this->myWindow;
		$mc = &windowContext($win->contextID(), "");			
		
		ob_start(); // write HTML first to a buffer
		
		//GGF3.2.2
		// tried also this: <div style="width:10000px; height:10000px; overflow:auto; "> to avoid word wrap in table but CSS works better
		echo '
'.$prefix.'<script type="text/javascript">';
		echo '
'.$prefix.'var exptime = new Date();';
		echo '
'.$prefix.'var increm = exptime.getTime() + 100000000;';
		echo '
'.$prefix.'exptime.setTime(increm);';
		echo '
'.$prefix.'var h = window.innerHeight != null? window.innerHeight: document.documentElement && document.documentElement.clientHeight ? document.documentElement.clientHeight:document.body != null? document.body.clientHeight:null;';
		echo '
'.$prefix.'document.cookie = "GGFPageHeight=" + h + "; expires=" + exptime.toGMTString();';
		echo '
'.$prefix.'</script>';
		// see $_COOKIE[PageHeight]
		echo '
'.$prefix.'<table  class="'.get_class($this).'" title="'.$this->tiptext.'"><tr>';
		if (!$this->deleteCallback == "") {
			echo "<td></td>"; // del column
		}
		if (!$this->viewCallback == "") {
			echo "<td></td>"; // view column
		}
		if (!$this->copyCallback == "") {
			echo "<td></td>"; // copy column
		}
		
		//GGF3.2
		// analyze sortcolumns
		$sorter = $this->myWindow->order() ;
		$criteria = $sorter->getCriteria();
		$i = 0;
		foreach ($criteria as $crit) {
			$i = $i+1;
			$chunk = explode(" ", $crit);
			$sortindex[$chunk[0]] = $i ;
			if (isset($chunk[1])) {
				$sortdir[$chunk[0]] = $chunk[1]; 	
			} else {
				$sortdir[$chunk[0]] = 'asc'; 	
			}
		}
				
		
		$i = 0;
		
		foreach ($this->externalColumnNames() as $col) {
				
			if (isset($sortindex[$this->columns[$i]])) { // col $i is a sort column
				$indicatorfile = "GGFIcons/GGF".$sortindex[$this->columns[$i]].$sortdir[$this->columns[$i]].".gif";
				$indicator = '<img src="'.$indicatorfile.'" width="31" height="18" border="0" title="sort order">';
			} else {
				$indicator = "";
			}

			$maxsize = $this->columnsize[$this->columns[$i]];
			if ($maxsize==0) {
				$maxsize = $maxColumnSize;
			}
			$coltitle = substr($col,0,$maxsize);
			
			if (!$this->sortCallback == "") {
				$sortURL = $sortpath."&control=".$this->name."&col=".$this->columns[$i];
				$mc->regCallbackURL(substr($sortURL,strpos($sortURL, "GGFDispatch")));
				echo  '<td><b><a href="'.$sortURL.'" title="sort by column '.$col.'" target="_top">'.$coltitle.$indicator.'</a></b></td>';
			} else {
				echo  "<td><b>".$coltitle.$indicator.'</b></td>';
			}
			$i = $i +1;
		}
		echo "</tr>";
		$zebra = TRUE;
		foreach($this->value as $row) {
		
			$updateURL = $updatepath."&pkvalue=".$this->myWindow->PKValue($row);
			$mc->regCallbackURL(substr($updateURL,strpos($updateURL, "GGFDispatch")));
			if (!$this->deleteCallback == "") {
				$deleteURL = $deletepath."&pkvalue=".$this->myWindow->PKValue($row);
				$mc->regCallbackURL(substr($deleteURL,strpos($deleteURL, "GGFDispatch")));
			}
			if (!$this->viewCallback == "") {
				$viewURL = $viewpath."&pkvalue=".$this->myWindow->PKValue($row);
				$mc->regCallbackURL(substr($viewURL,strpos($viewURL, "GGFDispatch")));
			}
			if (!$this->copyCallback == "") {
				$copyURL = $copypath."&pkvalue=".$this->myWindow->PKValue($row);
				$mc->regCallbackURL(substr($copyURL,strpos($copyURL, "GGFDispatch")));
			}
			// the javascript here is not mandatory, but helps to mark all columns at once
			if ($zebra) {
				echo '
'.$prefix.'<tr  class="'.get_class($this).'LN1" '/*onClick="javascript:top.document.location.href = \''.$updateURL.'\';" */.'onMouseOver="this.bgColor=\'#d09999\';this.color=\'#FFFFFF\';" onMouseOut="this.bgColor=\'#FFFFAF\';this.color=\'#000000\';">';
				$zebra = FALSE;
			} else {
				$zebra = TRUE;
				echo '
'.$prefix.'<tr class="'.get_class($this).'LN2" '/*onClick="javascript:top.document.location.href = \''.$updateURL.'\';" */.'onMouseOver="this.bgColor=\'#d09999\';this.color=\'#FFFFFF\';" onMouseOut="this.bgColor=\'#FFFFFF\';this.color=\'#000000\';">';
			}
			
			if (!$this->deleteCallback == "") {
				echo '<td bgcolor="#FFFFFF"><a href="'.$deleteURL.'" target="_top"><img src="GGFIcons/GGFdelete.gif" width="16" height="16" border="0" title="Delete row"></a></td>';
			}
			if (!$this->viewCallback == "") {
				echo '<td bgcolor="#FFFFFF"><a href="'.$viewURL.'" target="'.$this->viewTarget.'"><img src="GGFIcons/GGFview.gif" width="16" height="16" border="0" title="View row"></a></td>';
			}
			if (!$this->copyCallback == "") {
				echo '<td bgcolor="#FFFFFF"><a href="'.$copyURL.'" target="_top"><img src="GGFIcons/GGFcopy.gif" width="16" height="16" border="0" title="Copy row"></a></td>';
			}
								
			foreach ($this->columns as $col) {
				$maxsize = $this->columnsize[$col];
				if ($maxsize==0) {
					$maxsize = $maxColumnSize;
				}
				echo '<td>';
				if ($zebra) {
					echo '<a href="'.$updateURL.'"  class="'.get_class($this).'TXT1" target="'.$this->updateTarget.'">'.substr($row[$col],0,$maxsize)."</a>";
				} else {
					echo '<a href="'.$updateURL.'" class="'.get_class($this).'TXT2" target="'.$this->updateTarget.'">'.substr($row[$col],0,$maxsize)."</a>";
				}
				echo '</td>';
			}
			echo "</tr>";		
		}
		echo '</table>';
		
		$html = ob_get_contents();
		ob_end_clean();
   		return $html;
	}
}



/**
 * models an entryfield in a form
 * 
 * This control must not be used in a window, only in a dialog.
 *
 * @todo version, test
 * @package GGF
 * @version 2.3
 * @since 2.3
 * @author Gerald Zincke, Austria
 * @copyright 2005 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.htm#top
 */
class  GGFEntryfield extends GGFControl {
	// this models a normal text entry field
	
//	protected $html;
//	protected $name;
//	protected $value;
//	protected $size;
//	protected $extraHTML;
	protected $maxlength;
	protected $autofocus=FALSE;
	protected $inputClass=1;
	
	function __construct($name, $value, $size, $maxlength=0, $tiptext="", $extraHTML="", $mandatory=false) {
		parent::__construct($name, $value, $size, $tiptext, $extraHTML, $mandatory);
		if ($maxlength == 0) {
			$this->maxlength = $size;
		} else {
			$this->maxlength = $maxlength;
		}
	}
	
	public function isEntry() {
		return TRUE;
	}

	public function isForWindow() {
		// tells, if the control may be used in a window
		return FALSE;
	}

	public function isReadonly() {
		return $this->readonly;
	}

	public function setReadWrite() {
		$this->readonly = FALSE;
	}
	
	public function setAutofocus($parm=TRUE) {
		$this->autofocus = $parm;
	}
	
	public function setInputClass($parm=0) {
		$this->inputClass = $parm;
	}
	
	public function inputClass() {
		return $this->inputClass;
	}
	
	protected function setHTML() {
		$this->html = '<input  class="'.get_class($this).'"  type="text"     name="%s"   value="%s" size="%d" maxlength="%d" title="%s" %s %s style="%s">';
	}

	public function processRequest($myContext) {
		
		global $errorStack;
		
		if (isset($_REQUEST[$this->name])) {
			// note: also readonly-controls will be read (for redisplay)
			//       recognize "readonly" in readControls
			//echo '<br><br>'.__CLASS__."->".__FUNCTION__.':setting '.$this->name;
			//debug_print_backtrace();
			$val = $_REQUEST[$this->name];
			
			if (isset($this->inputClass)) {
				if ($this->mandatory || ($val >"")) {
					if ($this->inputClass==0) {
						// no check
					} elseif (GGFControlDialog::$inputPattern[$this->inputClass][0] == "#") {
						if (!preg_match(GGFControlDialog::$inputPattern[$this->inputClass], $val)) {
							//good
						} else {
							//$val = htmlspecialchars($val);; not needed. done at output
							$errorStack->pushUsrMsg(1,'Input: "'.$val.'" for '.$this->name.' not accepted as '.GGFControlDialog::$inputPatternText[$this->inputClass]);
						}
					} else {
						if (preg_match(GGFControlDialog::$inputPattern[$this->inputClass], $val)) {
							// good
						} else {
							//$val = htmlspecialchars($val); not needed. done at output
							$errorStack->pushUsrMsg(1,'Input: "'.$val.'" for '.$this->name.' not accepted as '.GGFControlDialog::$inputPatternText[$this->inputClass]);
						}
					}
				}
			}
			$this->value = $val;			
		}
	}

	public function render($demo=FALSE,$index=1,$indent=0) {
		$prefix = str_pad("",1+$indent*3);
		$ro = "";
		if ($this->readonly) {
			$ro =' readonly ';
		} 
		$af = "";
		if ($this->autofocus) {
			$af = ' autofocus ';
		} 
		$extra = $this->controlStyle();
		$extra = $extra.$this->extraHTML;
		
		$postfix = '';
		if ($this->mandatory) {
			$postfix = '*';
		}
	 	if ($this->autofocus) {
			$postfix = $postfix.sprintf('<script> if (!("autofocus" in document.createElement("input"))) { document.getElementsByName("%s")[0].focus();} </script>',$this->name);
		}
		
		//if ($this->size > 0) {
		//	$extra = sprintf(' style="width:%dpx; overflow:hidden" ', $this->size).$extra;
		//}
		return $prefix.sprintf($this->html, $this->name, htmlspecialchars($this->value), $this->size, $this->maxlength, $this->tiptext, $af, $ro, $extra).$postfix;
	}
}

/**
 * models an entryfield for a file upload
 * 
 * This control must not be used in a window, only in a dialog.
 *
 * @todo check if hidden input is needed
 * @package GGF
 * @version 2.3
 * @since 2.3
 * @author Gerald Zincke, Austria
 * @copyright 2005 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.htm#top
 */
class GGFFileEntryfield extends GGFEntryfield {

//	protected $html;
//	protected $name;
//	protected $value;
//	protected $size;
//	protected $extraHTML;
//	protected $maxlength;
//	protected $readonly=FALSE; not used here
	protected $accept;


	function __construct($name, $value, $size, $accept="text/*", $filemaxlength=0, $tiptext="", $extraHTML="") {
		parent::__construct($name, $value, $size, $filemaxlength, $tiptext, $extraHTML);
		$this->accept = $accept;
	}

	public function isForWindow() {
		// tells, that the control must not be used in a window
		return FALSE;
	}

	protected function setHTML() {
		// <input type="hidden" name="MAX_FILE_SIZE" value="30000">
		$this->html = '<input  class="'.get_class($this).'" type="file" name="%s" value="%s" 
		size="%d" accept="%s" title="%s" %s %s style="%s">';
	}

	public function processRequest($myContext) {
		// @todo: check if this needs rework
		if (isset($_REQUEST[$this->name])) {
			$this->value = $_REQUEST[$this->name];
		}
	}

	public function render($demo=FALSE,$index=1,$indent=0) {
		$prefix = str_pad("",1+$indent*3);
		
		$extra = $this->controlStyle();
		$ml = "";
		if ($this->maxlength > 0) { // this is the maximum file size, not all browsers check this
			$ml =' maxlength="'.$this->maxlength.'" ';
		}
		$af = "";
		if ($this->autofocus) {
			$af = ' autofocus ';
		} 
		
		$extra = $extra.$this->extraHTML;
		
		$postfix = '';
		if ($this->mandatory) {
			$postfix = '*';
		}
		// @todo: check if that makes really sense here
	 	if ($this->autofocus) {
			$postfix = $postfix.sprintf('<script> if (!("autofocus" in document.createElement("input"))) { document.getElementsByName("%s")[0].focus();} </script>',$this->name);
		}
		
		return $prefix.sprintf($this->html, $this->name, $this->value, $this->size, $this->accept, $this->tiptext, $af, $ml, $extra).$postfix;
	}
}

/**
 * models a password entryfield
 * 
 * This control must not be used in a window, only in a dialog.
 *
 * @todo version, test
 * @package GGF
 * @version 2.3
 * @since 2.3
 * @author Gerald Zincke, Austria
 * @copyright 2005 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.htm#top
 */
class  GGFPasswordEntryfield extends GGFEntryfield  {

	protected function setHTML() {
		$this->html = '<input  class="'.get_class($this).'"  type="password" name="%s"   value="%s" size="%d" maxlength="%d" title="%s" %s>';
	}
	
	public function isEntry() {
		return TRUE;
	}

}

/**
 * models a multi-line entry field in a form
 * 
 * This control must not be used in a window, only in a dialog.
 *
 * @todo version, test
 * @package GGF
 * @version 2.3
 * @since 2.3
 * @author Gerald Zincke, Austria
 * @copyright 2005 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.htm#top
 */
class  GGFMultilineEntryfield extends GGFEntryfield  {

//	protected $html;
//	protected $name;
//	protected $value;
//	protected $size;
//	protected $extraHTML;
//	protected $maxlength;
	protected $vsize;

	function __construct($name, $value, $size, $vsize=3, $maxlength=0, $tiptext="", $extraHTML="") {
		parent::__construct($name, $value, $size, $maxlength, $tiptext, $extraHTML);
		$this->vsize = $vsize;
		if ($maxlength == 0) {
			$this->maxlength = $size * $vsize;
		} else {
			$this->maxlength = $maxlength;
		}
	}

	protected function setHTML() {
		$this->html = '<textarea  class="'.get_class($this).'" name="%s" cols="%d" rows="%d" maxlength="%d" title="%s" %s %s style="%s">';
	}
	public function isEntry() {
		return TRUE;
	}
	
	public function render($demo=FALSE,$index=1,$indent=0) {
		$prefix = str_pad("",1+$indent*3);
		
		$extra = $this->controlStyle();
		
		$ro = "";
		if ($this->readonly) {
			$ro =' readonly ';
		} 
		
		$af ="";
		if ($this->autofocus) {
			$af =' autofocus ';
		} 
		
		$extra = $extra.$this->extraHTML;
		
		$html = sprintf($this->html, $this->name, $this->size, $this->vsize, $this->maxlength, $this->tiptext, $af, $ro, $extra);
		$html = $html.$this->value.'</textarea>';
		
		$postfix = '';
		if ($this->mandatory) {
			$postfix = '*';
		}
	 	if ($this->autofocus) {
			$postfix = $postfix.sprintf('<script> if (!("autofocus" in document.createElement("input"))) { document.getElementsByName("%s")[0].focus();} </script>',$this->name);
		}
		
		return $prefix.$html.$postfix;
	}	
}

//---the Buttons--------------------------------------------------------
/**
 * models a normal pushbutton
 *
 * This control must not be used in a window, only in a dialog.
 * 
 * @todo version, test
 * @package GGF
 * @version 2.3
 * @since 2.3
 * @author Gerald Zincke, Austria
 * @copyright 2005 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.htm#top
 */
class  GGFPushbutton extends GGFControl {

//	protected $html;
//	protected $name;
//	protected $value;
//	protected $size;
//	protected $extraHTML;
	protected $callback;
	protected $validator; // Note: this function will be called during rendering and during callback processing


	function __construct($name, $value, $callback, $validator="", $size=0, $tiptext="", $extraHTML="") {
		parent::__construct($name, $value, $size, $extraHTML);
		$this->callback = $callback;
		$this->validator = $validator;
		$this->tiptext = $tiptext;
	}

	public function isForWindow() {
		// tells, that the control must not be used in a window
		return FALSE;
	}

	protected function setHTML() {
		$this->html = '<input  class="'.get_class($this).'" type="submit" name="%s" value="%s" title="%s" %s style="%s">';
	}
	public function callback() {
		return $this->callback;
	}

	public function setCallback($aMethod) {
		$this->callback = $aMethod;
	}
	
	public function isButton() { // tells if its a form submit button
		return TRUE;
	}
	protected function prepareRegName($demo, $win, $val) {
		$mc = &windowContext($win->contextID(), "");
		
		if (($val=="") || (GGFControlWindow::$staticCallbacks)) {
			if ($demo) {
				// modify name to prevent calling callBack-Function in GGFControlWindow::processFormEvent
				$myName = "Demo:".$myName;
			} else {
				$myName = $this->name;
				$mc->regCallbackURL("Button:".$myName);					
			}
		} else { // use validator
		
			if ($demo) {
				// modify name to prevent calling callBack-Function in GGFControlWindow::processFormEvent
				$myName = "Demo:".$myName;
			} else {
				if (GGFControlWindow::$staticCallbacks) {
					// create static URL for HTML-only Form generation 
					$myName = $this->name;
				} else {
					if ($win->$val()) { //$validator must be the name of a function returning true when the button should be active
						$myName = $this->name;
						$mc->regCallbackURL("Button:".$myName);					
					} else {
						$this->disabled = TRUE;
						$myName = "disabled: ".$myName;
					}
				}
			}
		}
		return $myName;
	}
	/*
	 * This function is called during from window processFormEvent
	 * (in demo mode this will not be called because the button is not rendered by its real name)
	 * to prevent call callbacks for inactive buttons
	 */
	public function isActive() { // tells if button is enabled
		$win = $this->myWindow;
		$val = $this->validator;

		if (!$val =="") {
			if ($win->$val()) { //$validator must be the name of a function returning true when the button should be active
				return TRUE;
			} else {
				return FALSE;
			}
		} else { // no validator
			return TRUE;
		}
	}
	public function value() {
		return $this->value;
	}
	
	public function render($demo=FALSE,$index=1,$indent=0) {
		$prefix = str_pad("",1+$indent*3);
	
		$win = $this->myWindow;
		$val = $this->validator;

		$extra = $this->controlStyle();
		
		if ($this->size > 0) {
			$extra = $extra.sprintf(' width:%dpx;  ',$this->size);
		} 
		$extra = $extra.$this->extraHTML;

		$myName = $this->prepareRegName($demo, $win, $val);
		if ($this->disabled) {
			$da = ' disabled ';
		} else {
			$da = "";
		}
		return $prefix.sprintf($this->html, $myName, $this->value,  $this->tiptext, $da, $extra);
	}
}

/**
 * models a pushbutton that is decorated with a gif or jpg picture
 *
 * This control must not be used in a window, only in a dialog.
 * 
 * @todo version, test
 * @package GGF
 * @version 2.3
 * @since 2.3
 * @author Gerald Zincke, Austria
 * @copyright 2005 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.htm#top
 */
class  GGFImagePushbutton extends GGFPushbutton {
//	protected $html;
//	protected $name;
//	protected $value;
//	protected $size;
//	protected $extraHTML;
//	protected $callback;
//	protected $validator;
	protected $imageURL;
	protected $valuex;
	protected $valuey;
	
	function __construct($name, $value, $imageURL, $callback="", $validator="", $size=100, $tiptext="", $extraHTML="") {
		parent::__construct($name, $value, $callback, $validator, $size, $tiptext, $extraHTML);
		// print_r($image);
		$this->imageURL = $imageURL;
	}

	public function isForWindow() {
		// tells, that the control must not be used in a window
		return FALSE;
	}

	function setImageURL($shape) {
		$this->imageURL = $shape;
	}

	protected function setHTML() {
		$this->html = '<input  class="'.get_class($this).'"  type="image" src="%s" name="%s" value="%s" title="%s" %s style="%s">';
	}

	public function processRequest($myContext) {
		if (isset($_REQUEST[$this->name.'.x'])) {
			$this->valuex = $_REQUEST[$this->name.'.x'];
		}
		if (isset($_REQUEST[$this->name.'.y'])) {
			$this->valuey = $_REQUEST[$this->name.'.y'];
		}
		if (isset($_REQUEST[$this->name.'_x'])) {
			$this->valuex = $_REQUEST[$this->name.'_x'];
		}
		if (isset($_REQUEST[$this->name.'_y'])) {
			$this->valuey = $_REQUEST[$this->name.'_y'];
		}
	}

	public function valuex() {
		return $this->valuex;
	}
	
	public function valuey() {
		return $this->valuey;
	}
	
	public function render($demo=FALSE,$index=1,$indent=0) {
		$prefix = str_pad("",1+$indent*3);
		
		// print_r($this);
		if ($this->size > 0) {
			$extra = sprintf(' width:%dpx; ', $this->size).$this->extraHTML;
		} else {
			$extra = $this->extraHTML;
		}
		
		$win = $this->myWindow;
		$val = $this->validator;

		$myName = $this->prepareRegName($demo, $win, $val); // prepare a registered button name
		if ($this->disabled) {
			$da = ' disabled ';
		} else {
			$da = "";
		}
		return $prefix.sprintf($this->html, $this->imageURL, $myName, $this->value, $this->tiptext, $da, $extra);
	}
}
//---the Selectors-------------------------------------------
/**
 * serves as a base class for selector controls (listbox...)
 *
 * The value of a Selection Control is an array and is for output of the displayed options only
 * getSelection() delivers the user's choice
 * - an array of integers with the index of the selected options (returnIndex is set to true)
 * - an array of strings with the selected options (values)
 * - an array of the keys of the selected options (if keys have been set)
 *
 * @abstract
 * @todo version, test
 * @package GGF
 * @version 3.2
 * @since 2.3
 * @author Gerald Zincke, Austria
 * @copyright 2005 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.htm#top
 */
class GGFSelectionControl extends GGFControl {
	// this is an abstract class
	
//	protected $html;
//	protected $name;
//	protected $value;
//	protected $size;
//	protected $extraHTML;
	protected $vsize;
	protected $selections; // subset of the above with keys starting from 0 = preselected values
	protected $keys; // keys for the shown values
	protected $pre=TRUE; //preformatted data

	protected $returnIndex; // if true, return a selection index instead of line value
				// in this case the $selections also must be numbers from 0

	function __construct($name, $options, $selections, $hsize=100, $vsize= 15, $tiptext="", $extraHTML = "", $returnIndex=FALSE, $mandatory=false, $pre=TRUE) {
		parent::__construct($name, $options, $hsize, $tiptext, $extraHTML, $mandatory);
		//echo "<br>options:";
		//print_r($this->value);
		$this->selections = $selections;
		if (is_array($this->selections[0])) {
   			xdebug_print_function_stack( 'Invalid selections.' );		
   		}
		
		$this->vsize = $vsize;
		$this->returnIndex = $returnIndex;
		$this->pre = $pre;
	}

	public function setPreformatted($pre=TRUE){
		$this->pre = $pre;
	}

	public function valueType(){
		return "array";
	}
	
	public function selectionsType(){
		return "array";
	}
	
	/**
   	 * get the currently selected/highlighted items
   	 *
   	 * @return array containing the values that are highlighted. Will be a single valued array for single selection controls
   	 * 
   	 */
	public function getSelection() {
		return $this->selections;
	}
	
	/**
   	 * set the items to be highlighted/selected
   	 *
   	 * @param array containing the values to be highlighted. Has to be a single valued array for single selection controls
   	 * 
   	 */
	public function setSelection($selections) {
		//error_log_adv('GGFSelectionControl.setSelection.'.print_r(xdebug_get_function_stack(),TRUE));
		$this->selections = unserialize(serialize($selections));
		//if (is_array($this->selections[0])) {
   		//	xdebug_print_function_stack( 'Invalid selections.' );		
   		//}
	}
	
	public function setKey($key) {
		$this->keys = $key;
	}
	//GGF3.2	
	public function key() {
		return $this->keys;
	}

	public function processRequest() {
		if (isset($_REQUEST[$this->name])) {
			$this->selections = $_REQUEST[$this->name];
			if (is_array($this->selections[0])) {
   				xdebug_print_function_stack( 'Invalid selections.' );		
   			}
			
		//GGF3.2
		} else { // workaround, to allow re-setting of checkboxes
			$this->selections = array();
		}
	}

	public function getSelectionIndex() {
		// note : does only work for unique entries
		// If you need a selection index, set returnIndex in the constructor
		$index = array();
		if (isset($this->keys)) {
			$val = $this->keys;
		} else {
			$val = $this->value;
		}
		foreach ($this->selections as $selitem) {
			array_push ($index, array_search ( $selitem, $val)); 
		}
		return $index;
	}

    /**
	 * this allows to fill a listbox etc. with columnar data
	 * Use the  function of the window instead
	 * 
	 * @deprecated 
	 */
	public function fillSelectionControlER($ERTypeName, $columns, $columnwidths = array(), $filter = null, $sorter = null, $addDefaultSorter = true, $valueFields = array()) {
 		return $this->myWindow->fillSelectionControlER($this->name, $ERTypeName, $columns, $columnwidths , $filter, $sorter, $addDefaultSorter, $valueFields);
 	}


 

}

	
/**
 * models a single-select listbox
 *
 * This control must be used in a window only if the selection is not 
 * important and it is only used to display information.
 * 
 * @todo version, test
 * @package GGF
 * @version 2.3
 * @since 2.3
 * @author Gerald Zincke, Austria
 * @copyright 2005 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.htm#top
 */
class GGFListbox extends GGFSelectionControl {

//	protected $html;
//	protected $name;
//	protected $value;
//	protected $size;
//	protected $extraHTML;
//	protected $vsize;
//	protected $selections; // subset of the above with keys starting from 0 = preselected values

	protected function setHTML() {
		$this->html = '<select  class="'.get_class($this).'" 	name="%s[]"  size="%s" title="%s" %s %s style="%s">';
	}

	//GGF3.3
	public function processRequest() {
		parent::processRequest();
		//$this->value = $this->selections[0];
	}

	public function render($demo=FALSE,$index=1,$indent=0) {
		//print_r($this);
		$prefix = str_pad("",1+$indent*3);
		
		//GGF3.1
		ob_start(); // write HTML first to a buffer
		
		$extra = $this->controlStyle();
		
		if ($this->size > 0) {
			$extra = $extra.sprintf(' width:%dpx; ', $this->size).$this->extraHTML;
			// $extra = sprintf(' style="max-width:%dpx; overflow:auto" ', $this->size).$this->extraHTML;
		
		} else {
			$extra = $extra.$this->extraHTML;
		}
		
		$ro = "";
		if ($this->readonly) {
			$ro = ' readonly ';
		}
		
		$da = "";
		if ($this->isDisabled) {
			$da = ' disabled ';
		}
		

		echo "
".$prefix;
		printf($this->html,$this->name, $this->vsize, $this->tiptext, $ro, $da, $extra);		
		
		$i = 0;
		//echo "<br>options:";
		$myOptions = $this->value;
		//ini_set('xdebug.collect_params', '4');
		
		foreach ($myOptions as $item) {
			if ($this->returnIndex) {
				$val = $i;
			} elseif((isset($this->keys[$i])) && ($this->keys[$i] > "")) {
				$val = $this->keys[$i]; 
			} else {
				$val = $item;
			}
			$i++;
			$sel = "";
			//error_log_adv('GGFListbox-render:'.print_r($this->selections,TRUE));
			//print_r($val);
			foreach ($this->selections as $selopt) {
				if (strcmp($val,$selopt)== 0) {
					$sel = 'selected="selected"';
				}
			}
			
			echo "
".$prefix;
			echo '<option '.$sel.' value="'.$val.'">';
			$txt = htmlspecialchars($item);
			if ($this->pre) {
				$txt = str_replace(" ", "&nbsp;", $txt);
			}
			echo  $txt.'</option>';
		}
		echo '</select>';
		if ($this->mandatory) {
			echo"*";
		}
		
		$html = ob_get_contents();
		ob_end_clean();
   		return $html;
	}
	
	// GGF rework - DG, 08.08.06:
	public function isSingleSelection() {
		return TRUE;
	}
}

/**
 * models a drop-down selector
 *
 * This control must be used in a window only if the selection is not 
 * important and it is only used to display information.
 * 
 * @todo version, test
 * @package GGF
 * @version 2.3
 * @since 2.3
 * @author Gerald Zincke, Austria
 * @copyright 2005 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.htm#top
 */
class GGFDropDown extends GGFListbox {

	function __construct($name, $options, $selection, $hsize=100, $extraHTML = "", $returnIndex=FALSE, $mandatory=false, $pre=FALSE) {
		parent::__construct($name, $options, $selection, $hsize, 1, '', $extraHTML, $returnIndex, $mandatory, $pre);
	}
}

/* this class is deprecated */
class GGFComboBox extends GGFDropDown {
	
}


/**
 * models a multiple-select listbox
 *
 * This control must be used in a window only if the selection is not 
 * important and it is only used to display information.
 * 
 * @todo version, test, seperate versions for windows and dialogs
 * @package GGF
 * @version 2.3
 * @since 2.3
 * @author Gerald Zincke, Austria
 * @copyright 2005 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.htm#top
 */
class GGFMultipleListbox extends GGFListbox {

	//GGF3.3
	public function processRequest() {
		parent::processRequest();
		//$this->value = $this->selections;
	}

	protected function setHTML() {
		$this->html = 
		'<select  class="'.get_class($this).'" 	name="%s[]" size="%s" multiple title="%s" %s %s style="%s">';
	}
	
	// GGF rework - DG, 08.08.06:
	public function isSingleSelection() {
		return FALSE;
	}
}

/**
 * models a vertical or horizontal radio button group
 *
 * getSelection() of a radio group delivers
 * - an integer with the index of the selected option (returnIndex is set to true)
 * - the value of the selected option (string)
 * - the key of the selected option (if keys is set) 
 *
 * This control must be used in a window only if it is only used to 
 * display information. (no user modifications are to be read in)
 * 
 * @todo version, test
 * @package GGF
 * @version 3.0
 * @since 2.3
 * @author Gerald Zincke, Austria
 * @copyright 2005 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.htm#top
 */
class GGFRadioGroup extends GGFSelectionControl {

	protected $groupBox;
	
	function __construct($name, $options, $selection, $hsize=100, $extraHTML = "", $returnIndex=FALSE, $verticalSize=1, $groupBox=TRUE, $tiptext="") {
		parent::__construct($name, $options, $selection, $hsize, $verticalSize, $tiptext, $extraHTML, $returnIndex, $mandatory=false);
		$this->groupBox = $groupBox;
	}
	
	public function selectionsType(){
		return "string";
	}
	
	public function processRequest() {
		if (isset($_REQUEST[$this->name])) {
			$this->selections = $_REQUEST[$this->name];
			if (is_array($this->selections[0])) {
   				xdebug_print_function_stack( 'Invalid selections.' );		
   			}
		} else { // suppress workaround, for of checkboxes
		}
	}

	public function render($demo=FALSE,$index=1,$indent=0) {
		$prefix = str_pad("",1+$indent*3);

		$extra = $this->controlStyle(). $this->extraHTML;
		if ($this->groupBox) {
			$html = '<fieldset  class="'.get_class($this).'" style="'.$extra.'" ><legend>'.$this->name.'</legend>';		
		} else {
			$html = '<div  class="'.get_class($this).'" style="'.$extra.'" >';		
		}
		$break = "";
		
		$i = 0;
		foreach ($this->value as $item) {
			$c = "";
			if ($this->returnIndex) {
				$val = $i;
				if ($val == ($this->selections)) {
					$c = " checked ";
				} 
				
			} elseif((isset($this->keys[$i])) && ($this->keys[$i] > "")) {
				$val = $this->keys[$i]; 
				if (strcmp($val,$this->selections)==0) {
					$c = " checked ";
				} 
				
			} else {
				$val = $item;
				if (strcmp($val,$this->selections)==0) {
					$c = " checked ";
				} 
			}
			
			//echo "<br>val ".$i."=".$val;
			$ro = "";
			if ($this->readonly) {
				$ro = ' readonly ';
			}
			if ($this->isDisabled) {
				$ro = ' disabled '.$ro;
			}
			
			$html = $html.$break."
".$prefix.				
				'<LABEL  class="'.get_class($this).'" FOR="'.$this->name.$i.'"   >'.
				'<input type="radio" '.$c.' name="'.$this->name.'" ID="'.$this->name.$i.
				'" value="'.$val.'" title="'.$this->tiptext.'" '.$ro.' >';
			 
			$txt = htmlspecialchars($item);
			if ($this->pre) {
				$txt = str_replace(" ", "&nbsp;", $txt);
			}
			 
			$html = $html. $txt. '</LABEL>';
			if ($this->vsize>1) {
				$break = '<br>';
			}
			$i++;
		}
		if ($this->groupBox) {
			$html = $html. '</fieldset>';
		} else {
			$html = $html. '</div>';
		}
		return $prefix.$html;
	}

	public function isSingleSelection() {
		return true;
	}
	
}
	 
/**
 * models a vertical or horizontal checkbox group
 *
 * getSelection() of a checkbox group delivers
 * - an indexed array with boolean values (if returnIndex is set to true)
 * - an associative array with the values as keys and a boolean as value
 * - an associative array with the set keys as keys and a boolean as value
 *
 * This control must be used in a window only if it is only used to 
 * display information. (no user modifications are to be read in)
 * 
 * @todo version, test
 * @package GGF
 * @version 3.1.1
 * @since 2.3
 * @author Gerald Zincke, Austria
 * @copyright 2005 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.htm#top
 */
class GGFCheckboxGroup extends GGFRadioGroup {

	public function render($demo=FALSE,$index=1,$indent=0) {
		$prefix = str_pad("",1+$indent*3);
		
		//GGF3.1.1
		$extra = $this->controlStyle(). $this->extraHTML;
		if ($this->groupBox) {
			$html = '<fieldset  class="'.get_class($this).'" style="'.$extra.'" ><legend>'.$this->name.'</legend>';		
		} else {
			$html = '<div  class="'.get_class($this).'" style="'.$extra.'" >';		
		}
		$break = "";
		
		$i = 0;
		foreach ($this->value as $option) {
			if ($this->returnIndex) {
				$val = $i;
			} elseif((isset($this->keys[$i])) && ($this->keys[$i] > "")) {
				$val = $this->keys[$i]; 
			} else {
				$val = $option;
			}
			
			$ro = "";
			if ($this->readonly) {
				$ro = ' readonly ';
			}
			
			$da = "";
			if ($this->isDisabled) {
				$da = ' disabled ';
			}
			
			//GGF3.1.1
			$c = "";
			if ($this->selections[$val]) {
					$c = " checked ";
			}
			
			$html = $html.$break."
".$prefix.								
				'<LABEL FOR="'.$this->name.$i.'" style="'.$extra.'" >'.
//				'<input type="checkbox" '.$c.' name="'.$this->name.'[]" ID="'.$this->name.$i.
				'<input type="checkbox" '.$c.$ro.$da.' name="'.$this->name.$i.'" ID="'.$this->name.$i.
			'" value="'.$val.'" title="'.$this->tiptext.'" style="'.$extra.'" >';
			 
			$txt = htmlspecialchars($option);
			if ($this->pre) {
				$txt = str_replace(" ", "&nbsp;", $txt);
			}
			$html = $html. $txt. '</LABEL>';
			if ($this->vsize>1) {
				$break = '<br>';
			}
			$i++;
		}
		if ($this->groupBox) {
			$html = $html. '</fieldset>';
		} else {
			$html = $html. '</div>';
		}
		return $prefix.$html;
	}

	public function isSingleSelection() {
		return FALSE;
	}
	
	public function processRequest() {
		$this->selections = array();
		for ($i = 0; $i < count($this->value); $i++) {
				if (isset($_REQUEST[$this->name.$i])) {
					if ($this->returnIndex) {
						$this->selections[] = TRUE;
					} elseif(isset($this->keys) && ($this->keys[$i] > "")) {
						$this->selections[$this->keys[$i]] = TRUE; 
					} else {
						$this->selections[$this->value[$i]] = TRUE;
					}
				} else {
					if ($this->returnIndex) {
						$this->selections[] = FALSE;
					} elseif(isset($this->keys) && ($this->keys[$i] > "")) {
						$this->selections[$this->keys[$i]] = FALSE; 
					} else {
						$this->selections[$this->value[$i]] = FALSE;
					}
				}
		}
		if (is_array($this->selections[0])) {
   			xdebug_print_function_stack( 'Invalid selections.' );		
   		}
		
	}
}

/**
 * models a single checkbox
 *
 * getSelection() of a checkbox delivers
 * - TRUE or FALSE (returnIndex and keys are ignored)
 *
 * This control must be used in a dialog to enter information: It can be 
 * used in a window, if it is only used to 
 * display information. (no user modifications are to be read in)
 * 
 * @todo version, test
 * @package GGF
 * @version 3.0
 * @since 3.0
 * @author Gerald Zincke, Austria
 * @copyright 2005 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.htm#top
 */
class GGFCheckbox extends GGFRadioGroup {

	function __construct($name, $label, $selected=FALSE, $hsize=0, $tiptext="" , $extraHTML = "") {
		parent::__construct($name, $label, $selected, $hsize, $vsize=1, $extraHTML, $returnIndex=FALSE);
		$this->tiptext =$tiptext;
	}
	
	/**
   	 * set the items to be highlighted/selected
   	 *
   	 * @param boolean TRUE: box is checked
   	 * 
   	 */
	public function setSelection($selections=TRUE) {
		$this->selections = $selections;
	}

	public function render($demo=FALSE,$index=1,$indent=0) {
		$prefix = str_pad("",1+$indent*3);
		
		$extra = $this->controlStyle()." ".$this->extraHTML;
		
		$ro = "";
		if ($this->readonly) {
			$ro = ' readonly ';
		}
		
		$da = "";
		if ($this->isDisabled) {
			$da = ' disabled ';
		}
		
		if ($this->selections) {
			$c = " checked ";
		} else {
			$c = "";
		}
			
		$html = '<LABEL FOR="'.$this->name.'"  class="'.get_class($this).'" style="'.$extra.'">'.
			'<input type="checkbox" '.$c.$ro.$da.' name="'.$this->name.'" ID="'.$this->name.
			'" value="1" title="'.$this->tiptext.'" >';
		 
		$txt = htmlspecialchars($this->value);
		if ($this->pre) {
			$txt = str_replace(" ", "&nbsp;", $txt);
		}
			
		$html = $html. $txt . '</LABEL>';
		
		return $prefix.$html;
	}

	public function valueType(){
		return "string";
	}
	
	public function selectionsType(){
		return "boolean";
	}
	
	public function processRequest() {

		if (isset($_REQUEST[$this->name])) {
			$this->selections = TRUE;
		} else {
			$this->selections = FALSE;
		}
	}
}

/**
 * models an element of a menu-bar 
 *
 * This is a hyperlink. Do not use this in dialogs that contain a form.
 * 
 * @todo version, test
 * @package GGF
 * @version 2.3
 * @since 2.3
 * @author Gerald Zincke, Austria
 * @copyright 2005 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.htm#top
 */
class GGFMenuItem extends GGFControl {

//	protected $html;
//	protected $name;
//	protected $value;
//	protected $size;
//	protected $extraHTML;
//	protected $tiptext="";
	protected $target;
	protected $validator;
	protected $callback;
	
	function __construct($name, $callback, $validator, $tiptext="", $target="_top",  $extraHTML="") {
		parent::__construct($name, $name, $size=0, $tiptext, $extraHTML);
		$this->target = $target;
		$this->validator = $validator;
		$this->callback = $callback;
	}

	protected function setHTML() {
		$this->html = '<a href="%s" class="'.get_class($this).'"  target="%s" title="%s" style="%s">&nbsp;%s&nbsp;</a>';
	}
	
	public function render($demo=FALSE,$index=1,$indent=0) {
		$prefix = str_pad("",1+$indent*3);
		
		//print_r($this);
		$win = $this->myWindow;
		$val = $this->validator;
		
		if (($val=="") || (GGFControlWindow::$staticCallbacks)) {
			// note: retrieving a callback URL will also declare it valid. So this must not be done for disabled menus
			if ($demo) {
				$url = $win->callbackURL("eventDemoClicked");
			} else {
				$url = $win->callbackURL($this->callback);
			}
		} else {
			//print_r($this);
			if ($demo) {
				$url = $win->callbackURL("eventDemoClicked");
			} else {
				if ($win->$val()) {
					// note: retrieving a callback URL will also declare it valid. So this must not be done for disabled menus
					$url = $win->callbackURL($this->callback);
				} else {
					// make placeholder
					//return $prefix.'<font class="GGFmenu" color=gray>&nbsp;'.$this->value.'&nbsp;</font>';
					return $prefix.'<font class="'.get_class($this).'" color=gray>&nbsp;'.$this->value.'&nbsp;</font>';				
				}
			}
		}
		return $prefix.sprintf($this->html, $url, $this->target, $this->tiptext, $this->extraHTML, $this->name);		
	}
}

//---Containers-----------------------------------------------------	
/**
 * This allows to iterate through the controls of a pane.
 * The iterator's next function returns the next control of the pane.
 * 
 * @todo version, test
 * @package GGF
 * @version 2.3
 * @since 2.3
 * @author Gerald Zincke, Austria
 * @copyright 2005 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.htm#top
 */
class GGFPaneIterator {
	protected $i;
	protected $arr;
	protected $iter;
	
	function __construct($controlsArray) {
		$this->arr = $controlsArray;
		$this->i = 0;
	}
	public function next() {
		global $errorStack;
		if ($this->i >= count($this->arr)) {
			return 0;
		}
		$j = 0;
		foreach($this->arr as $k => $v) {
		   	if ($j == $this->i) {
		   		$c = $v;
		   		break;
		   	}
		   	$j++;
		}
		//$errorStack->pushMsg($this->i.". examine ".$c->name());
		if ($c->isPane()) {
			//$errorStack->pushMsg($c->name()." is pane, goto sub");
			if ($this->iter == 0) {
				$this->iter = $c->paneIterator();
			} 
			$res = $this->iter->next();
			if ($res == 0) { // nothing more in sub-iterator
				$this->iter = 0;
				$this->i++;
				return $this->next();
			} else {
				return $res;
			}
		} else { // normal control
			// $errorStack->pushMsg($c->name()." is control");
			$this->i++;
			return $c;
		}
	}
}
/**
 * models a placeholder that does nothing
 * 
 * @todo version, test
 * @package GGF
 * @version 
 * @since 
 * @author Gerald Zincke, Austria
 * @copyright 2011 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.htm#top
 */
class  GGFEndPane extends GGFControl {
		
	function __construct($name) {
		//name may also be "", tiptext is here for future use
		parent::__construct($name, "", 0, "", "");
	}
		
	protected function setHTML() {
		$this->html = '';
	}

	public function processRequest($ignore) {
		// nothing to do
	}

	public function setMenuIndex($ignore) {
		// nothing to do
	}
	
	public function menuIndex() {
		return '';
	}
	
	public function render($demo=FALSE,$index=1,$indent=0) {
		
		return '';
	}
}


/**
 * concrete base class for containers for controls. Every window or dialog
 * must have at least one pane. panes can be nested. 
 * 
 * @todo version, test
 * @package GGF
 * @version 4.1
 * @since 2.3
 * @author Gerald Zincke, Austria
 * @copyright 2005 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.htm#top
 */
class GGFControlPane {
	// models a set of controls, that make up the contents of a window
	
	protected $name;
	protected $myContainer; // if nested
	protected $myWindow;
	protected $controls = array();
	protected $disabled = FALSE;
	protected $inTable = FALSE;
	protected $tableLevel = 0;
	public    $tabindex = 0; // tbi
	protected $bodyExtraHTML = "";
	protected $frameID=0;
	protected $paneType;
	protected $isFrameset=FALSE;
	protected $style = '';
	/** @var string RRGGBB string for foreground color. Empty string: default */
	protected $Foreground="";
	
	/** @var string RRGGBB string for background color. Empty string: default */
	protected $Background="";
	
	/** @var string font family specification. default: same as parent */
	protected $fontFamily="";

	/** @var string font size specification including unit. default: same as parent */
	protected $fontSize="";
	
	/** @var array font style options set. default: same as parent */
	protected $fontStyle=array(); 
	
	/** @var array font style options set. default: same as parent */
	protected $lineHeight="";
	
	
	function __construct($aWindow,$name="") {
		$this->myWindow = $aWindow;
		$this->name = $name;
		$this->paneType='span';
	}
	
	public function myClass() {
		return get_class($this);
	}

	/**
	 * 
	 * @deprecated
	 * 
	 */
	public function initializeProperties($pValues){
		$v = array();
		$r = array();
		foreach ($pValues as $key => $value) {
			if ($value == "") {
				$r[] = $key;
			} else {
				$v[$key] = $value;
			}
		}
		$this->resetP($r);
		$this->initP($v);
	}
	/**
	 * 
	 * resets property values of the control
	 * @param array $props array with instance variable names
	 */
	public function resetP($props){
		foreach ($props as $pName) {
			if (!(in_array($pName, array('html', 'html2', 'myWindow', 'controls', 'paneType')))) {
				if ($pName == "colSizes") {
					$this->$pName = array();
				} else {
					$this->$pName = "";
				}	
			}
		}
	}
	
	/**
	 * 
	 * initialize property values of the control
	 * @param array $pValues array with instance variable names => values
	 */
	public function initP($pValues){
		foreach ($pValues as $pName => $pValue) {
			if (!(in_array($pName, array('html', 'html2', 'myWindow', 'controls', 'paneType')))) {
				if ($pName == "colSizes") {
					$this->$pName = explode("|", $pValue);
				} else {
					$this->$pName = $pValue;
				}
			}
		}
	}
	
	
	public function isFrame() {
		return (!($this->frameID == 0));
	}
	public function isFrameset() {
		return $this->isFrameset;
	}
	//GGF 2.3
	public function isMenu() {
		return FALSE;
	}

	public function setIsFrameset($parm=TRUE) {
		$this->isFrameset=$parm;
	}
	public function frameID() {
		return $this->frameID;
	}
	public function setFrameID($id) {
		$this->frameID = $id;
	}

	protected function controlStyle() {
		$style=" ";
		if ($this->Background > "") {
			$style=$style."background-color:".$this->Background.";";
		}
		if ($this->Foreground > "") {
			$style=$style."color:".$this->Foreground.";";
		}
		
		if ($this->fontFamily > "") {
			$style = $style.' font-family:'.$this->fontFamily."; ";
		}

		if (isset($this->fontStyle["bold"])) {
			if ($this->fontStyle["bold"]) {
				$style = $style.' font-weight:bold; ';
			}
		}
		if (isset($this->fontStyle["italic"])) {
			if ($this->fontStyle["italic"]) {
				$style = $style.' font-style:italic; ';
			}
		}
		if (isset($this->fontStyle["underline"])) {
			if ($this->fontStyle["underline"]) {
				$style = $style.' text-decoration:underline; ';
			}
		}	
		
		if ($this->lineHeight > "") {
			$style = $style.' line-height:'.$this->lineHeight."; ";
		}
		if ($this->fontSize > "") {
			$style = $style.' font-size:'.$this->fontSize."; ";
		}
		
		return $style;
	}
	
	public function paneIterator() {
		return new GGFPaneIterator($this->controls);
	}

	public function isEntry() {
		return FALSE;
	}

	public function isReadonly() {
		return FALSE;
	}

	public function isButton() { // tells if its a form submit button
		return FALSE;
	}	

	// GGF rework - DG, 08.08.06:
	public function isSingleSelection() {
		return FALSE;
	}

	public function isPane() {
		return TRUE;
	}
	
	
	public function add($aGGFControl) {
	
		//debug_print_backtrace();			
		array_push($this->controls,$aGGFControl);
		$aGGFControl->setContainer($this);
		$aGGFControl->setWindow($this->myWindow);
			if ($this->myWindow == 0) {
				error_log_adv(__FILE__.': Tried to add a control to a pane that has no window');
				// error_log_adv(debug_print_backtrace(TRUE));
			}
		
		$aGGFControl->setTabindex(count($this->controls)-1);
		$this->tabindex++;		
	}
	
	/*
	* remove a control given by name rom pane
	*/
	public function delete($aName) {
		$control = $this->named($aName);
		if ($control >0) {
			$cont = $control->container();
			if ($cont == $this) {
				$new = array();
				foreach ($this->controls as $c) {
					if (strcmp($c->name(),$aName)==0) {
						//echo "<br>delete:".$aName;
					} else {
						//echo "<br>take:  ".$c->name();
						array_push($new,$c);
					}
				}
				$this->controls = $new;
				$this->tabindex--;
			} else {
				$cont->delete($aName);
			}
		}
		
	}
	
	/*
	 * remove all previously added controls from the pane
	 */
	public function clearControls() {
		$this->controls = array();
		$this->inTable = FALSE;
		$this->tabindex = 0; // tbi
	}
	
	/**
	 * add a pane as a frame.
	 * Note: the containing window then must consist of a frameset definition
	 * This can be created by adding a single generic control to the window.
	 * Sample: 
	 * $this->pane->add(new GGFGenericControl("mainfs", '
	 *		<frameset cols="500,*">  
	 *		  <frame src="'.$mc->frameCallbackURL(1).'" name="headerPane" marginwidth="0" marginheight="0" frameborder="0" framespacing="0" border="0" scrolling="yes" >
	 *		  <frame src="'.$mc->frameCallbackURL(2).'" name="scrollPane" marginwidth="0" marginheight="0"  frameborder="0" framespacing="0" border="0">
	 *		  <noframes>Your Web-Browser cannot show frames. Use a modern browser.</noframes>
	 *		</frameset>'));
	 * 
	 * @param type $frameID
	 * @param type $aGGFControlPane
	 * @return type 
	 */
	public function addFrame($frameID, $aGGFControlPane) {
		// add the pane and declare it a frame
		// this will be rendered into framebuffer $frameID
		// note: panes of the same window must not share the same frameID
		$aGGFControlPane->setFrameID($frameID);
		return $this->add($aGGFControlPane);
	}
	
	public function setTabindex($i) {
		$this->tabindex = $i;
	}
	public function setBodyExtraHTML($html) {
		$this->bodyExtraHTML = $html;
	}
	public function bodyExtraHTML() {
		return $this->bodyExtraHTML;
	}
	public function name() {
		return $this->name;
	}
	public function newSpace($nr=1) {
		$html = '';
		for ($i=1;$i <= $nr;$i++) {
			$html = $html.'&nbsp';
		} 
		$this->add(new GGFGenericControl("newSpace".$nr,$html));		
	}
	public function newLine() {
		$this->add(new GGFGenericControl("newLine","<br>"));		
	}
	public function newP() {
		$this->add(new GGFGenericControl("newP","<p>"));		
	}

	public function newTable($aGGFControl,$cellAttributes='',$rowAttributes='',$tableAttributes=' border-width:0px; ') {
		if ($this->inTable ) {
			$this->add(new GGFGenericControl('closeTable','</td></tr></table>'));
		}
		$this->add(new GGFGenericControl('_newTable',
		'<table class="GGFControlTable" style="'.$tableAttributes.'" ><tr class="GGFCTRow" style="'.$rowAttributes.'"><td class="GGFCTCell" style="'.$cellAttributes.'" >'));
		$this->inTable = TRUE;
		$this->tableLevel++;
		$this->add($aGGFControl);
	}
	public function newRow($aGGFControl,$cellAttributes='',$rowAttributes='',$tableAttributes=' border-width:0px; ') {
		if ($this->inTable ) {
			$this->add(new GGFGenericControl('_newRow','</td></tr><tr class="GGFCTRow" style="'.$rowAttributes.'"><td class="GGFCTCell" style="'.$cellAttributes.'">'));		
		} else {
			$this->add(new GGFGenericControl('_newTable',
			'<table class="GGFControlTable" style="'.$tableAttributes.'"><tr class="GGFCTRow" style="'.$rowAttributes.'" ><td  class="GGFCTCell" style="'.$cellAttributes.'">'));
			$this->tableLevel++;
		}
		$this->inTable = TRUE;
		$this->add($aGGFControl);
	}
	public function newCell($aGGFControl,$cellAttributes='',$rowAttributes='',$tableAttributes=' border-width:0px; ') {
		if ($this->inTable ) {
			$this->add(new GGFGenericControl('_newCell','</td><td class="GGFCTCell" style="'.$cellAttributes.'" >'));		
		} else {
			$this->add(new GGFGenericControl('_newTable','<table class="GGFControlTable" style="'.$tableAttributes.'" ><tr class="GGFCTRow" style="'.$rowAttributes.'"><td class="GGFCTCell" style="'.$cellAttributes.'">'));
			$this->tableLevel++;
		}
		$this->inTable = TRUE;
		$this->add($aGGFControl);
	}

	public function closeTable() {
		if ($this->inTable ) {
			$this->add(new GGFGenericControl('_closeTable','</td></tr></table>'));		
			$this->tableLevel--;
		} 
		$this->inTable = FALSE;
	}
	
	public function setContainer($cont) {
		$this->myContainer = $cont;
	}
	
	public function setWindow($win) {
		$this->myWindow = $win;
	}
	
	public function setAutofocus() {
	}
	
	protected function doNamed($name) {
		if ($this->name == $name) {
			return $this;
		}
		foreach($this->controls as $c) {
			if ($c->name() == $name) {
				return $c;
			} elseif ($c->isPane()) {
				$d = $c->doNamed($name); // recursive
				if (!$d == 0) {
					return $d;
				} 
			}
		}
		return 0;
	}
	
	public function named($name) {
		global $errorStack;
		$c = $this->doNamed($name);
		if (!$c == 0) {
			return $c;
		}
		//$errorStack->pushMsg("Control named ".$name." not found in ".$this->name.".");
		return 0;
	}
	
	public function myWindow() {
		return $this->myWindow;
	}
	
	public function setDisabled($on = TRUE) {
		$this->disabled = $on;
		foreach ($this->controls as $c) {
			$c->setDisabled($on);
		}
	}
	
	public function processRequest($mc) {
		foreach ($this->controls as $c) {
			$c->processRequest($mc);
		} 
	}
	protected function style() {
		return $this->style;
	}
	public function setStyle($aStyleSpec) {
		$this->style = $aStyleSpec;
	}
	
	public function render($demo=FALSE,$index=1,$indent=0) {
		$prefix = str_pad("",1+$indent*3);
		
		global $traceLevel;
		
		$html = "
".$prefix."<!---begin pane:".$this->name."-->".
				"
".$prefix."<".$this->paneType.'  id="'.$this->name.'" style="'.$this->style().$this->controlStyle().'" >';
		foreach($this->controls as $control) {
			$controlHTML = $control->render($demo,$index,$indent+1);
		   	if ($traceLevel>=3) { error_log_adv ("$this->appname ". get_class($this)."->".__FUNCTION__.": rendering:".$control->name());}
 				
			if ($control->isFrame()) {
				// error_log_adv("processing frame: ".$control->name());
				$mc = &windowContext($this->myWindow->contextID(),0);
				if ($control->isFrameset()) {
					$framepre = '<html><head><title>a Frame</title></head>';
					$framepost = '
</html>';
				} else {
					$framepre = '<html><head><link rel="stylesheet" type="text/css" href="GGFFormats.css"></head>'.
					'<body class="'.$this->myWindow->bodyClass().'" style="'.$control->bodyExtraHTML().'" >';
					$framepost = '
</body>
</html>';
				}
				// write rendering text into frame buffer, don't add to $html
					
				$mc->frames[$control->frameID()] = $framepre.$controlHTML.$framepost;
 				$mc->save();			
			} else {
		   		// if ($traceLevel>=2) { error_log_adv ("$this->appname ".__CLASS__."->".__FUNCTION__.": rendering:".$control->name());}
				$html = $html.'
'.$controlHTML;
			}
		}
		/* if ($this->inTable) {
			$html = $html.'
'.$prefix.'</td></tr></table>';
		}*/
		while ($this->tableLevel>0) {
			$html = $html.'
'.$prefix.'</td></tr></table>';
			$this->tableLevel--;
		}
		
		return $prefix.$html.
		"
".$prefix."</".$this->paneType.">".
		"
".$prefix."<!---end pane:".$this->name."-->";
	}

	protected function controlID() {
		// this should be unique per window. t.b.d.
	}	
	public function editRender() {
		//return $this->render(); // cannot edit framesets
		if ($this->isFrameset) {
			return $this->render(); // cannot edit framesets
		} else {
			return '<fieldset><legend><b>'.$this->name.'</b></legend>
			<table><tr>
			<td><table>
			   <tr><td><a href="'.$this->myWindow->callbackURL("_eventEditControl").'&_controlID='.$this->controlID().'">e</a><td></tr>
			   <tr><td><a href="'.$this->myWindow->callbackURL("_eventDeleteControl").'&_controlID='.$this->controlID().'">d</a><td></tr>
			</table></td>
			<td>'.$this->render().'</td>
			</tr></table></fieldset>';
		}
	}
}

/**
 * models the system tray, containing some standard hyperlinks to be found on any 
 * window or dialog. 
 *
 * @todo version, test
 * @package GGF
 * @version 2.3
 * @since 2.3
 * @author Gerald Zincke, Austria
 * @copyright 2005 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.htm#top
 */
class GGFSystemTrayPane extends GGFControlPane{
	// this is used by Dialog windows
	public function myClass() {
		return get_class($this);
	}
	
	protected function style() {
		//return 'style="text-align:right; border-width:1px; border-color:#696969;"';
		return ' position:absolute; right:5px; ';
	}
}

/**
 * models a pane that starts on a new line and is tagged as DIV.  
 *
 * @todo version, test
 * @package GGF
 * @version 4.0
 * @since 4.0
 * @author Gerald Zincke, Austria
 * @copyright 2011 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.htm#top
 */
class GGFNewLinePane extends GGFControlPane {
	///** @var integer the horizontal size of the control */
	//protected $size;
	
	

	public function render($demo=FALSE,$index=1,$indent=0) {
		$this->paneType = "DIV";
		return parent::render($demo,$index,$indent);
	}
}

/**
 * models a pane that is enclosed in a fieldset box. 
 *
 * @todo version, test
 * @package GGF
 * @version 3.0
 * @since 3.0
 * @author Gerald Zincke, Austria
 * @copyright 2005 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.htm#top
 */
class GGFFieldsetPane extends GGFControlPane {

	public function render($demo=FALSE,$index=1,$indent=0) {
		$prefix = str_pad("",1+$indent*3);
		
		return $prefix.'<fieldset  class="'.get_class($this).'"  style="'.$this->style().$this->controlStyle().'" ><legend>'.$this->name.'</legend>
		'.parent::render($demo, $index, $indent+1).'
'.$prefix.'</fieldset>';
	}
}

/**
 * models a form. This is an area in a dialog where input controls
 * can be located. 
 *
 * @todo version, test
 * @package GGF
 * @version 2.3
 * @since 2.3
 * @author Gerald Zincke, Austria
 * @copyright 2005 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.htm#top
 */
class GGFFormPane extends GGFControlPane {
	
//	protected $myContainer;
//	protected $myWindow;
//	protected $controls = array();
//	protected $disabled = FALSE;
//	public $tabindex = 0;
	protected $html;
	protected $demohtml;
	
	protected $demohtml2;
	protected $html2;
	protected $formScriptName="processForm.pl";
	
	
	function __construct($aWindow,$name="") {
		parent::__construct($aWindow,$name);
		$this->setHTML();
		$this->setHTML2();
	}
	
	public function isForWindow() {
		// tells, that the control must not be used in a window
		// todo: check this.
		return FALSE;
	}

	public function myClass() {
		return get_class($this);
	}
	
	protected function setHTML() {
		$this->html ='%s<!---Form:'.$this->name.'-->
%s<form method="POST" class="'.$this->myClass().'" style="%s" action="%s"  enctype="multipart/form-data" target="_top" >';
		
		$this->demohtml ='<!---Form:'.$this->name.'-(demohtml)--><div class="'.$this->myClass().'" style="%s"  >';
	}
	
	protected function style() {
		// return '';
		return $this->style;
	}
		
	protected function setHTML2() {
		// this Javascript takes care that a specific  control gets the focus
		$this->html2 = '%s<!---end form:%s-->
%s</form>
%s<script type="text/javascript"><!--
%sif (entry=document.getElementById("ffocus")) {
%s    entry.focus(); //or entry.select();
%s}
%s//--></script>';
		$this->demohtml2 = ' ';
	}
	
	public function render($demo=FALSE,$index=1,$indent=0) {
		$prefix = str_pad("",1+$indent*3);
		
		$this->setHTML();
		$this->setHTML2();
		
		if ($demo) {
			$url = $this->myWindow->callbackURL("eventDemoClicked");
			$html = sprintf($this->demohtml,$this->style().$this->controlStyle());
		} else {
			if (GGFControlWindow::$staticCallbacks) {
				$url = $this->myWindow->callbackURL($this->formScriptName);
			} else {
				$url = $this->myWindow->callbackURL("processFormEvent");
			}
			$html = sprintf($this->html,$prefix,$prefix,$this->style().$this->controlStyle(),$url);
			
		}
	
		//print_r($html);
		foreach($this->controls as $control) {
			$controlHTML = $control->render($demo,$index,$indent+1);
 			
			global $traceLevel;
		   	if ($traceLevel>=3) { error_log_adv ("$this->appname ".__CLASS__."->".__FUNCTION__.": rendering:".$control->name());}
			$html = $html.'
'. $controlHTML;
		}
		while ($this->tableLevel>0) {
			$html = $html.'
'.$prefix.'</td></tr></table>';
			$this->tableLevel--;
		}
		
		if ($demo) {
			return $html.$this->demohtml2;	
		} else {
			return $html.'
'.sprintf($this->html2,$prefix,$this->name,$prefix,$prefix,$prefix,$prefix,$prefix,$prefix);
				
			
		}
	}
}

/**
 * FormPane with style used in dialogs 
 *
 * @todo version, test
 * @package GGF
 * @version 2.3
 * @since 2.3
 * @author Gerald Zincke, Austria
 * @copyright 2005 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.htm#top
 */
class GGFDialogFormPane extends GGFFormPane {
	// this is used by Dialog windows
	/*protected function style() {
		return 'style="border-width:4px; background-color:#C0C0C0; border-color:#696969; 
			border-style:outset; padding:5px; width:99%"';
	}*/

	public function myClass() {
		return 'GGFDialogFormPane';
	}
	
	
	public function isForWindow() {
		// tells, that the control must not be used in a window
		return FALSE;
	}


}

/**
 * Form with style used in the Note Window 
 *
 * @todo version, test
 * @package GGF
 * @version 2.3
 * @since 2.3
 * @author Gerald Zincke, Austria
 * @copyright 2005 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.htm#top
 */
class GGFNotePane extends GGFFormPane{
	// this is used by Dialog windows
	
	public function myClass() {
		return get_class($this);
	}
	
	protected function style() {
		return 'border-width:4px;  padding:5px; ';
	}

}

	
/**
 * Pane to hold menu-items. This must not be used in dialogs. 
 *
 * @todo version, test
 * @package GGF
 * @version 2.3
 * @since 2.3
 * @author Gerald Zincke, Austria
 * @copyright 2005 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.htm#top
 */
class GGFMenu extends GGFControlPane {
	// note: Menus must not be added into form panes, except they are located in
	// a frame and the entry-fields are in another frame
	protected $html;
	protected $html2;
	
	function __construct($aWindow,$name="") {
		parent::__construct($aWindow,$name);
		$this->setHTML();
		$this->setHTML2();		
	}

	public function isForDialogs() {
		// tells, that the control must not be used in a dialog
		return FALSE;
	}


	protected function setHTML() {
		$this->html = '<!---begin menu: '.$this->name.'-->
   <table bgcolor="#E0E0E0" border=0 width="100%">
   <tr><td>
   <table  class="'.get_class($this).'"  width="100%"  style="'.$this->style().$this->controlStyle().'" >
   <tr><td>
   <table>
   <tr>
';
	}

	protected function setHTML2() {
		$this->html2 = '<!---end menu: '.$this->name.' --></TR></table></TD><td></td></TR></table></td></Tr></table>';
	}
	
	//GGF 2.3
	protected function isActive() {
		if ($this->myWindow->selectedMenu() == 0) {
			// per default the top-level menu is active
			return !$this->myContainer->isMenu();
		} else {
			return ($this->myWindow->selectedMenu() == $this->name);	
		}
	}
	
	protected function isSubmenu() {
		return ($this->myWindow->subMenu() == $this->name);	
	}

	public function render($demo=FALSE,$index=1,$indent=0) {
		$prefix = str_pad("",1+$indent*3);
		

		if (!$this->isActive) {
			// iterate to the active menu
	
			$itr = $this->paneIterator();

			$c = $itr->next();
			while (!$c == 0) {
				if ($c instanceof GGFMenu) {
					if ($c->isActive()) {
						return $prefix.$c->render($demo,$index,$indent);
					}
				}
				$c = $itr->next();
			}
			//error_log_adv ($this->appname.__CLASS__."->".__FUNCTION__.": programming error: no active menu found.");
		}
		// we get here if the menu is active (or no other active menu found);
		$submenu = 0;
		
		if (count($this->controls) > 0) {
			$html = "";
			foreach ($this->controls as $item) {
				global $traceLevel;
			   	if ($traceLevel>=3) { error_log_adv ($this->appname.__CLASS__."->".__FUNCTION__.": rendering:".$item->name());}

				if ($c instanceof GGFMenu) { // submenu
					if ($item->isSubmenu()) {
						$submenu = $item;
					}
				} else {
					$controlHTML = $item->render($demo,$index,$indent);
 					
			   		$html = $html.
			   		'
'.$prefix.'<td>'.$controlHTML.
			   		'
'.$prefix.'</td>';
				}	
			}
			$html = $this->html.$html.'
'.$prefix.$this->html2;

			if (!$submenu == 0) {
				$controlHTML = $submenu->render($demo,$index,$indent);
 				$html = $html.'<br>'.$controlHTML;
			}
			
			return $prefix.$html;
		} else {
			return ""; // empty menu
		}
	}
	
}

/**
 * Pane to hold menu-items and sub-Menus. This must be used in wimdows and dialogs. 
 *
 * @todo version, test
 * @package GGF
 * @version 1.0
 * @since 4.0
 * @author Gerald Zincke, Austria
 * @copyright 2012 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.htm#top
 */
class GGFHMenu extends GGFControlPane {
	// note: Menus must not be added into form panes, except they are located in
	// a frame and the entry-fields are in another frame
	protected $value;
	protected $tiptext;
	protected $menuIndex;
	
	function __construct($aWindow,$name="",$value="", $tiptext="") {
		parent::__construct($aWindow,$name);
		$this->value = $value;
		$this->tiptext = $tiptext;
		
	}

	public function isForDialogs() {
		// tells, that the control can be used in a dialog
		return TRUE;
	}


	
	public function isActive() { // tells if button is enabled
		$win = $this->myWindow;
		$val = $this->validator;

		if (!$val =="") {
			if ($win->$val()) { //$validator must be the name of a function returning true when the button should be active
				return TRUE;
			} else {
				return FALSE;
			}
		} else { // no validator
			return TRUE;
		}
	}
	
	public function isButton() { // tells if its rendered to a submit button
		$win = $this->myWindow;
		return (is_subclass_of( $win , 'GGFControlDialog' ));
	}
	
	/**
         * @todo: check if needed
         */
	protected function isSubmenu() {
		return ($this->myWindow->subMenu() == $this->name);	
	}

	public function isMenu() {
		return TRUE;
	}
	
	public function menuIndex() {
		return $this->menuIndex;
	}
	/**
	 * sets the menuindex for the menu
	 * @param string indexString a string containing two bytes per menu-level for an index
	 */
	public function setMenuIndex($indexString) {
		$this->menuIndex = $indexString;
	}

	// @todo: check if button name is changed for demo mode
	public function render($demo=FALSE,$index=1,$indent=0, $menuLevel=0, $parentWidth=0) {
		global $traceLevel;
		global $HMItemHeight;
		$prefix = str_pad("",1+$indent*3);

		$html = array();
		if (!$this->myContainer->isMenu()) { // is top menu
        	$html[] = $prefix.'<!---begin menu: '.$this->name.'-->';
		} else {
		}
		// add menu title
		
		$win = $this->myWindow;
		$val = $this->validator;

		if ($this->value>"") {
			//$html[] = get_class($win); 
			if (is_subclass_of( $win , 'GGFControlDialog' )) {
				
				$myName = "_openHMenu".$this->menuIndex ;				
				// if ($demo) openung menu is allowed
				$mc = &windowContext($win->contextID(), "");
				$mc->regCallbackURL("Button:".$myName);					
				$da = "";
	
				if (!GGFControlWindow::$staticCallbacks) {
					if (!$val =="") {	
						if ($win->$val()) { //$validator must be the name of a function returning true when the button should be active
						} else {
							// make placeholder
							$myName = "disabled: ".$myName;
							$da = ' disabled ';
						}
					}
				}
				
				$button = $prefix.'<input  class="'.get_class($this).'" type="submit" name="%s" value="%s..." title="%s" '.$da.' style="%s">';
				$html[] = sprintf($button, $myName, $this->value, $this->tiptext, $this->extraHTML);		
			} else {
				$url = $win->callbackURL("_openHMenu".$this->menuIndex);
				$link = $prefix.'<a href="'.$url.'" class="'.get_class($this).'"  title="%s" style="%s">%s...</a>';
				
				if (($val=="") || (GGFControlWindow::$staticCallbacks)) {
					$html[] = sprintf($link, $this->tiptext, ' width:'.strlen($this->value).'em; '.$this->extraHTML, $this->value);		
				} else {
					if ($win->$val()) { //$validator must be the name of a function returning true when the button should be active
						$html[] = sprintf($link, $this->tiptext, ' width:'.strlen($this->value).'em; '.$this->extraHTML, $this->value);		
					} else {
						// make placeholder
						$html[] = '<span class="'.get_class($this).'" >&nbsp;'.$this->value.'&nbsp;</span>';				
					}
				}
			}
		}

		$nritems = count($this->controls);
		$i = 1;
		if ($nritems > 0) {
			$classdef = '';
			if (!$this->myContainer->isMenu()) {
				$classdef = 'class="'.get_class($this).'" id=GGFHMenu ';
			}
				
			$topDist=0; // Distance of menuitem from top (in em)
			$topInc= $HMItemHeight;
			$html[] = $prefix.'<ul '.$classdef.' >';
			foreach ($this->controls as $item) {
				if ($traceLevel>=3) { error_log_adv ($this->appname.__CLASS__."->".__FUNCTION__.": rendering:".$item->name());}

				$item->setMenuIndex($this->menuIndex.substr('0'.$i,-2));
				$controlHTML =  $item->render($demo,$i,$indent+1, $menuLevel+1 );
				$style = ' style="';
				$i++;
				if ($i == $nritems) {
					if ($this->myContainer->isMenu()) {
						$style = $style.'border-bottom-style:outset; ';
					}
				}
				//$html[] = "<!--".$item->name()." menuIndex: ".$item->menuIndex()." selected:".$win->selectedMenu()."-->";
				if ($item->menuIndex() == substr($win->selectedMenu(),0, strlen($item->menuIndex()))) {
					$liClass = ' class="menuOpen" ';
				} else {
					$liClass = '';
				}	
				
				// for 2nd flyouts (4th menu level, menu-index of items has 8 chars) 
				// the current CSS 3/2012 needs setting of height
				if (strlen($item->menuIndex()) > 7) {
					$style = $style.' top:'.$topDist.'em; ';
					$topDist = $topDist+$topInc;		
				} 
				$style = $style.'" ';
				
				
				if ($controlHTML>"") {
					$html[] = $prefix.'<li '.$liClass.' '.$style.'>';
					$html[] = $controlHTML;				
					$html[] = $prefix.'</li>';					
				}
			}	
			$html[] = $prefix.'</ul>';
		}
		if (!$this->myContainer->isMenu()) { // is the top menu
			$html[] = $prefix.'<!---end menu: '.$this->name.' -->';
        }
		return implode('
',$html);	
	}
	
}

/**
 * models an element of a menu in a hover menu 
 *
 * In windows this is rendered as a hyperlink. 
 * In dialogs, it is rendered to a pushbutton.
 * 
 * @todo version, test
 * @package GGF
 * @version 2.3
 * @since 4.0
 * @author Gerald Zincke, Austria
 * @copyright 2012 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.htm#top
 */
class GGFHMItem extends GGFControl {

//	protected $html;
//	protected $name;
//	protected $value;
//	protected $size;
//	protected $extraHTML;
//	protected $tiptext="";
	protected $target;
	protected $validator;
	protected $callback;
	protected $menuIndex;
	
	function __construct($name, $value, $callback, $validator="", $tiptext="", $target="_top",  $extraHTML="") {
		parent::__construct($name,$value,$size=0, $tiptext, $extraHTML="");		
		$this->target = $target;
		$this->validator = $validator;
		$this->callback = $callback;
	}

	public function isActive() { // tells if menu item is enabled
		$win = $this->myWindow;
		$val = $this->validator;

		if (!$val =="") {
			if ($win->$val()) { //$validator must be the name of a function returning true when the button should be active
				return TRUE;
			} else {
				return FALSE;
			}
		} else { // no validator
			return TRUE;
		}
	}
	
	public function isButton() { // tells if its rendered to a submit button
		$win = $this->myWindow;
		return (is_subclass_of( $win , 'GGFControlDialog' ));
	}
	
	public function callback() {
		return $this->callback;
	}

	protected function setHTML() {
		$this->html = '<a href="%s" class="'.get_class($this).'"  target="%s" title="%s" style="%s">%s</a>';
	}
	
	public function menuIndex() {
		return $this->menuIndex;
	}
	/**
	 * sets the menuindex for the menu
	 * @param string indexString a string containing two bytes per menu-level for an index
	 */
	public function setMenuIndex($indexString) {
		$this->menuIndex = $indexString;
	}
	
	public function render($demo=FALSE,$index=1,$indent=0) {
		$prefix = str_pad("",1+$indent*3);
		$win = $this->myWindow;
		$val = $this->validator;
		
		if (is_subclass_of ( $win , 'GGFControlDialog' )) { // menu in Dialog
			// render to input submit tags
			$mc = &windowContext($win->contextID(), "");
			$da = "";
			
			if (($val=="") || (GGFControlWindow::$staticCallbacks)) {
				$myName = $this->name;
				if ($demo) {
					// modify name to prevent calling callBack-Function in GGFControlWindow::processFormEvent
					$myName = "Demo:".$myName;
				} else {
					$mc->regCallbackURL("Button:".$myName);					
				}
			} else { // use validator
				if ($demo) {
					// modify name to prevent calling callBack-Function in GGFControlWindow::processFormEvent
					$myName = "Demo:".$myName;
				} else {
					if ($win->$val()) { //$validator must be the name of a function returning true when the button should be active
						$myName = $this->name;
						$mc->regCallbackURL("Button:".$myName);					
					} else {
						// make placeholder
						//return $this->value;
						//return $prefix.sprintf('<span %s>%s</span>',$extra,htmlspecialchars($this->value));
						$da = ' disabled ';
					}
				}
			}
			$button = '<input  class="'.get_class($this).'" type="submit" name="%s" value="%s" title="%s"  style="%s">';
			return $prefix.sprintf($button, $myName, $this->value, $this->tiptext, $this->extraHTML);		
		} else {
			// render to hyperlinks
			if (($val=="") || (GGFControlWindow::$staticCallbacks)){ // do not call validator for HTML-only Form generation 
				// note: retrieving a callback URL will also declare it valid. So this must not be done for disabled menus
				if ($demo) {
					$url = $win->callbackURL("eventDemoClicked");
				} else {
					$url = $win->callbackURL($this->callback);
				}
			} else {
				//print_r($this);
				if ($demo) {
					$url = $win->callbackURL("eventDemoClicked");
				} else {
										
					if ($win->$val()) {
						// note: retrieving a callback URL will also declare it valid. So this must not be done for disabled menus
						$url = $win->callbackURL($this->callback);
					} else {
						// make placeholder
						return $prefix.'<span class="'.get_class($this).'" >&nbsp;'.$this->value.'&nbsp;</span>';				
					}
				}
			}
			return $prefix.sprintf($this->html, $url, $this->target, $this->tiptext, ' width:'.strlen($this->value).'em; '.$this->extraHTML, $this->value);		
		}	
	}
}

/**
 * Pane to scroll a table.
 * @todo version, test
 * @package GGF
 * @version 4.0
 * @since 4.0
 * @author Gerald Zincke, Austria
 * @copyright 2011 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.htm#top
 */
class GGFScrollTablePane extends GGFControlPane {
	/* inherited variables
	protected $name;
	protected $myContainer; // if nested
	protected $myWindow;
	protected $controls = array();
	protected $disabled = FALSE;
	protected $inTable = FALSE;
	public $tabindex = 0; // tbi
	protected $bodyExtraHTML = "";
	protected $frameID=0;
	protected $paneType;
	protected $isFrameset=FALSE;
	protected $style = '';
	protected $Foreground="";
	protected $Background="";
	protected $fontFamily="";
	protected $fontSize="";
	protected $fontStyle=array(); 
	protected $lineHeight=""; 
	*/
	protected $html;
	protected $html2;
	
	protected $upCallback;
	protected $dnCallback;
	
	protected $size=0;
	protected $colSizes=array();
	
	function __construct($aWindow,$name="") {
		parent::__construct($aWindow,$name);
		$this->setHTML();
		$this->setHTML2();		
	}

	public function isForDialogs() {
		// tells, that the control can be used in a dialog
		return TRUE;
	}


	protected function setHTML() {
		$this->html = '%s<!---begin ScrollTable: '.$this->name.'-->
		<table class="'.get_class($this).'"  style="width:%s; %s ">
			<tr class="GGFSCTRow" >
				<td class="GGFSCTCell" >
	 				<table class="GGFSCTleft" style="width:100##; border:0px; height=100##; " >
	 					<tr class="GGFSCTRowleft" style="width:100##; height=100##;">
	 						<td class="GGFSCTCellleft" style="width:100##;" >
	 							<table class="GGFSCTData" style="" >';

		/* row
			 <!--- this is the headeritem -->
			   <tr class="GGFSCTDRow" style="width:100%;" >
				   <td  class="GGFSCTDCell" >
				   <span  class="GGFStaticfield"  style="">Nr</span>
				   </td>
				   <td  class="GGFSCTDCell" >
				   <span  class="GGFStaticfield"  style="">header</span>
				   </td>
			   </tr>
		*/
		
	}

	protected function setHTML2() {
		$this->html2 = '%s		</table><!---end data table-->
							</td>
							<td class="GGFSCTCellright" style="width:20px; height:100##;" >
	          					<span style=" height:100##;" >
	          					<table class="GGFSCTScroll" style="background-color:gray; width:20px; height:100##;">
				 					<tr class="GGFSCTSphr" style="background-color:black; height:5##;">
				 						<td class="GGFSCTSphc" >
	             							<span  class="GGFSCTSph">_</span>
	             						</td>
				 					</tr>
				 					<tr class="GGFSCTSupr" style="height:50##;">
				 						<td class="GGFPseudoPushbutton">
				 							<a    href="%s"  title="scroll up" target="_self" style="  overflow:hidden  "><b>^<br>|</b></a>			 
	             						</td>
				 					</tr>
				 					<tr class="GGFSCTSdnr" style="height:50##;">
				 						<td class="GGFPseudoPushbutton">
				 							<a    href="%s"  title="scroll down" target="_self" style="  overflow:hidden  "><b>|<br>v</b></a>
								        </td>
	             					</tr>
	             				</table>
	          					</span>
	       					</td>
       					</tr>
       				</table>
    			</td>
    		</tr>
    	</table>
	 <!--- end Scrolltable -->';
			
	}

	public function render($demo=FALSE,$index=1,$indent=0) {
		$prefix = str_pad("",1+$indent*3);
		
		$this->setHTML();
		$this->setHTML2();
		
		if ($demo) {
			$up = $this->myWindow->callbackURL("eventDemoClicked");
			$dn = $this->myWindow->callbackURL("eventDemoClicked");
		} else {
			if (GGFControlWindow::$staticCallbacks) {
				$up = $this->upCallback;
				$dn = $this->dnCallback;
			} else {
				$up = $this->myWindow->callbackURL($this->upCallback);
				$dn = $this->myWindow->callbackURL($this->dnCallback);
			}
		}
		
		if ($this->size == 0) {
			$this->size = "100%";
		} 
		
		$cg = '<colgroup>';
		foreach ($this->colSizes as $colSize) {
			$cg = $cg.'<col width="'.$colSize.'">';
		}
		$cg = $cg.'</colgroup>';
		
		
		$html = sprintf($this->html,$prefix,$this->size,$this->style().$this->controlStyle()).$cg;
		$html = str_replace("##", "%", $html);
		$i = 0;
		$controlHTML = array();
		
		foreach($this->controls as $control) {
			//print_r($control->name());
			if (($i == 0) and  (strcmp($control->name(),"_newTable") == 0)) {
				//echo "<br>".$i.": table ignored.".htmlspecialchars($control->render($demo,$index,$indent+1));
				$controlHTML[0]='<tr class="GGFCTRow" style="width:100%;" ><td  class="GGFSCTDCell" >'; // ignore
			} else { // begins with row as it should
				//echo "<br>".$i.": normal.".htmlspecialchars($control->render($demo,$index,$indent+1));
				$controlHTML[$i] = $control->render($demo,$index,$indent+1);
			}
			$i++;
		}

		// now that code will contain wrong table-classes
		for ($i = 0; $i < count($controlHTML); $i++) {
			if (strpos($controlHTML[$i],'class="GGFCTRow"')) {
				// color
				//echo "<br>color source:".$this->controls[$i+1]->name();
				$controlHTML[$i] = str_replace('class="GGFCTRow"', 'class="GGFSCTDRow" bgcolor="'.
				$this->controls[$i+1]->Background.'" ', $controlHTML[$i]);
			} elseif (strpos($controlHTML[$i],'class="GGFCTCell"')) {
				$controlHTML[$i] = str_replace('class="GGFCTCell"', 'class="GGFSCTDCell" ', $controlHTML[$i]);
			}
		}
		
		
		$html = $html.'
'.$prefix.implode("
",$controlHTML);
		
		$html2 = sprintf($this->html2,$prefix,$up,$dn,$prefix);		
		$html2 = str_replace("##", "%", $html2);
		
		return $html.$html2;	
	}
}
			

/**
 * models a text area representing items in a tree hierarchy.
 * This is an abstract class. Methods nrChildren and render have to be overwritten.
 * 
 * @todo version, test
 * @package GGF
 * @version 2.3
 * @since 2.3
 * @author Gerald Zincke, Austria
 * @copyright 2005 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.htm#top
 */
class GGFTreeListArea extends GGFControl {
	
	protected $updateCallback;
	protected $deleteCallback;
	protected $expandCallback;
	protected $collapseCallback;
	protected $collapseAllCallback;
	protected $cutCallback;
	protected $pasteCallback;
	protected $caption;
	protected $expanded;
	protected $cut;
	protected $mode;
	protected $selections;
	
	function __construct($name="aGGFTreeListArea", $caption="", $value=array(), 
		$updateCallback="eventUpdate", $deleteCallback="eventDelete",
		$expandCallback="eventExpand", $collapseCallback="eventCollapse", $collapseAllCallback="",
		$cutCallback="", $pasteCallback="", $tiptext="", $extraHTML="") {
		parent::__construct($name, $value, /*Size*/ 0, $tiptext, $extraHTML);
		$this->updateCallback=$updateCallback;
		$this->deleteCallback=$deleteCallback;
		$this->expandCallback=$expandCallback;
		$this->collapseCallback=$collapseCallback;
		$this->collapseAllCallback=$collapseAllCallback;
		$this->cutCallback=$cutCallback;
		$this->pasteCallback=$pasteCallback;
		$this->caption=$caption;
		$this->expanded = array();
		$this->cut = 0;
		$this->mode = 0;
		$this->selections = array();
		// $value = $rows;
	}
	
	public function isStyleable() {
		return FALSE;
	}

	public function setExpanded($expanded) {
		$this->expanded = $expanded;
	}
	
	public function setCut($element) {
		$this->cut = $element;
	}
	
	public function setMode($mode) {
		$this->mode = $mode;
	}

	/*
	 * Note: this method must be overwritten in derived classes.
	 * This is a demo implementation
	 */
	protected function nrChildren($element) {
		global $db;
		global $errorStack;
		$sql = "SELECT (COUNT(*) - 1) as 'childCount' FROM feature";
		$elements = array();
		for ($i = 0; $i <= 9; $i++) {
			if ($element['h'.$i] > 0) {
				array_push($elements, 'h'.$i.'='.$element['h'.$i]);
			} else {
				if (($i + 1) <= 9) {
					array_push($elements, 'h'.($i + 1).'='.$element['h'.($i + 1)]);
				}
				break;
			}
		}
		if (count($elements) > 0) {
			$sql .= ' WHERE '.implode(' AND ', $elements);
		}
				
		$res = $db->execSQL($sql);
		if (!$db->OK()) {
			return 0;
		} else {
			$row = $db->fetchRow($res);
			return $row['childCount'];
		}
	}
	
	public function setSelection($sel) {
		$this->selections = $sel;
	}
	
	public function getSelection() {
		if ($this->mode == 3) {
			return $this->selections;			
		} else {
			return array();
		}
	}

	/**
	 * @TODO: review if workaround for non-existing $_REQUEST value is OK
	 */
	public function processRequest() {
		if (isset($_REQUEST[$this->name])) {
			$this->selections = $_REQUEST[$this->name];
		} else {
			$this->selections = array();
		}
	}

	/** 
	 * render the control to HTML
	 * Note: this method must be overwritten in derived classes.
	 * This is a demo implementation
	 */
	public function render($demo=FALSE,$index=1,$indent=0) {
		$prefix = str_pad("",1+$indent*3);
		
		
		/*
		protected $updateCallback;
	protected $deleteCallback;
	protected $expandCallback;
	protected $collapseCallback;
	protected $collapseAllCallback;
	protected $cutCallback;
	protected $pasteCallback;
		*/
		
		ob_start(); // write HTML first to a buffer
		
		// head
		echo /*'<fieldset><legend><b>'.$caption.'</b></legend>*/'<table bordersize=0><tr><td><table>';
		echo '<tr><td>'/*<img src="GGFIcons/GGFcut.gif" width="16" height="16" border="0"></td><td><a href="" title="paste element/subtree as child"><img src="GGFpaste.gif" width="16" height="16" border="0"></a></td><td><img src="GGFIcons/GGFprev.jpg" width="10" height="10" border="0"></td><td><a href="" title="expand subtree"><img src="GGFIcons/plus.gif" width="10" height="10" border="0"></a></td><td> </td></tr>'*/;
		foreach ($this->value as $row) { // elements
			echo '<tr>';
			
			if ($this->mode == 1) {
				if ($this->cut > 0) {
					echo '<td><a href="'.$this->myWindow->callbackURL($this->cutCallback)."&pkvalue=".$this->myWindow->PKValue($row).'" title="cut this element/subtree" target="_top"><img src="';
					if ($this->cut == $row['Feature_ID']) {
						echo 'GGFIcons/GGFcut-red.gif" width="16" height="16" border="0"></a></td><td><a href="'.$this->myWindow->callbackURL($this->pasteCallback)."&pkvalue=".$this->myWindow->PKValue($row).'" title="paste element/subtree as child" target="_top"><img src="GGFIcons/GGFpaste.gif" width="16" height="16" border="0"></a></td>';
					} else {
						echo 'GGFIcons/GGFcut.gif" width="16" height="16" border="0"></a></td><td><a href="'.$this->myWindow->callbackURL($this->pasteCallback)."&pkvalue=".$this->myWindow->PKValue($row).'" title="paste element/subtree as child" target="_top"><img src="GGFIcons/GGFpaste.gif" width="16" height="16" border="0"></a></td>';
					}
				} else {
					echo '<td><a href="'.$this->myWindow->callbackURL($this->cutCallback)."&pkvalue=".$this->myWindow->PKValue($row).'" title="cut this element/subtree" target="_top"><img src="GGFIcons/GGFcut.gif" width="16" height="16" border="0"></a></td><td><img src="GGFIcons/GGFpaste-grey.gif" width="16" height="16" border="0" title="paste element/subtree as child (cut a element first)"></td>';
				}
				echo '<td><a href="'.$this->myWindow->callbackURL($this->deleteCallback)."&pkvalue=".$this->myWindow->PKValue($row).'" title="delete this element/subtree" target="_top"><img src="GGFIcons/GGFdelete.gif" width="16" height="16" border="0"></a></td>';
			} else if ($this->mode == 2) {
				//echo '<td><button type="submit" name="_save" value="save" title="re-sort elements like set in the entryfields"><img src="GGFIcons/GGFsave.gif" width="10" height="10" border="0" alt="save"></button></td>';
				echo '<td><input type="submit" style="background: url(GGFIcons/GGFsave.gif); height:16px; width:16px; " name="" value=""></td>';
			} else if ($this->mode == 3) {
				//echo '<td><button type="submit" name="_save" value="save" title="re-sort elements like set in the entryfields"><img src="GGFIcons/GGFsave.gif" width="10" height="10" border="0" alt="save"></button></td>';
				echo '<td><input type="checkbox" '.$c.' name="'.$this->name.'[]" ID="'.$this->name.$row['Feature_ID'].
			'" value="'.$row['Feature_ID'].'" title="'.$this->tiptext.'" '.$extra.'></td>';
			}

			echo '<td>';
			
			$children = $this->nrChildren($row);
			if ($children > 0) {
				if (isset($this->expanded[$row['Feature_ID']])) {
					echo /*'<a href="'.$this->myWindow->callbackURL($this->collapseAllCallback)."&pkvalue=".$this->myWindow->PKValue($row).'" title="collapse all subtrees"><img src="GGFIcons/back.gif" width="10" height="10" border="0"></a></td><td>*/
					'<a href="'.$this->myWindow->callbackURL($this->collapseCallback)."&pkvalue=".$this->myWindow->PKValue($row).'" title="collapse subtree" target="_top"><img src="GGFIcons/GGFminus.gif" width="10" height="10" border="0"></a>';
				} else {
					echo /*'<img src="GGFIcons/back.gif" width="10" height="10" border="0"></td><td>*/
					'<a href="'.$this->myWindow->callbackURL($this->expandCallback)."&pkvalue=".$this->myWindow->PKValue($row).'" title="expand subtree ('.$children.' nodes)" target="_top"><img src="GGFIcons/GGFplus.gif" width="10" height="10" border="0"></a>';
				}
			//} else {
				//echo /*'<img src="GGFIcons/back-grey.gif" width="10" height="10" border="0"></td><td>*/'<img src="GGFIcons/GGFplus-grey.gif" width="10" height="10" border="0">';
			}
			
			echo '</td><td>';
			for ($i = 0; $i <= 9; $i++) {
				if ($row['h'.$i] > 0) {
					if (($row['h'.($i + 1)] > 0) || ($this->mode != 2)) {
						echo '<input type="text" value="'.$row['h'.$i].'" size="2" maxlength="2" style="text-align:right;background-color:#DDDDDD;color:#222222;" readonly >';
					} else {
						echo '<input type="text" name="h';
						
						for($j = 0; $j <= $i; $j++) {
							echo '_'.$row['h'.$j];
						}
						
						echo '" value="'.$row['h'.$i].'" size="2" maxlength="2" style="text-align:right;">';
					}
				} else {
					break;
				}
			}
			echo ' <a href="'.$this->myWindow->callbackURL($this->updateCallback)."&pkvalue=".$this->myWindow->PKValue($row).'" title="'.$row['Description'].'" class="GGFtree" target="_top">'.$row['Name']."</a></td></tr>";
		}
		
		// tail
		echo '</table></td></tr></table>'/*</fieldset>'*/;
		$html = ob_get_contents();
		ob_end_clean();
   		return $prefix.$html;
	}
}

/**
 * models a pane that is enclosed in a scroling-divider. 
 *
 * @todo version, test
 * @package GGF
 * @version 3.0
 * @since 3.0
 * @author Gerald Zincke, Austria
 * @copyright 2005 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.htm#top
 */
class GGFScrollPane extends GGFControlPane {

	protected $width;
	protected $height;
	
	function __construct($aWindow, $width=900, $height=450, $name="") {
		$this->myWindow = $aWindow;
		$this->name = $name;
		$this->width = $width;
		$this->height = $height;
	}
	
	public function isStyleable() {
		return FALSE;
	}

	public function render($demo=FALSE,$index=1,$indent=0) {
		$prefix = str_pad("",1+$indent*3);
		
		return $prefix.'<div style="width:'.$this->width.'px; height:'.$this->height.'px; overflow:auto; ">'.
		/*.$this->style().'><legend><b>'.$this->name.'</b></legend>
		'*/
		'
'.$prefix.parent::render($demo,$index,$indent).
		'
'.$prefix.'</div>';
	}
}

/**
 * models an entryfield in a form which allows the selection of an existing
 * value via a 
 * 
 * This control must not be used in a window, only in a dialog.
 *
 * @todo test
 * @package GGF
 * @version 3.5.1-so
 * @since 3.5.1-so
 * @author Simon Opelt
 */
class  GGFSelectionEntryfield extends GGFEntryfield {
	
	protected $selectMessage = '- select -';
	protected $selections = array();
	
	function __construct($name, $value, $size, $maxlength=0, $tiptext="", $extraHTML="", $mandatory=false) {
		parent::__construct($name, $value, $size, $tiptext, $extraHTML, $mandatory);
		if ($maxlength == 0) {
			$this->maxlength = $size;
		} else {
			$this->maxlength = $maxlength;
		}
	}
	
	public function selectionsType(){
		return "array";
	}

	public function isForWindow() {
		// tells, that the control must not be used in a window
		return FALSE;
	}

	public function setSelectionValues($selections) {
		$this->selections = $selections;
	}
	
	public function valueType(){
		return "string";
	}
	
	protected function setHTML() {
		$this->html = '<fieldset class="'.get_class($this).'"><input   type="text" name="%s" value="%s" size="%d"'.
			' maxlength="%d" title="%s" %s %s><br><select name="%s_selection[]" 
		 size="%s" title="a non-blank value choosen here will be taken as input for the previous entry-field">';
	}

	public function processRequest($myContext) {
		if (isset($_REQUEST[$this->name])) {
			// note: also readonly-controls will be read (for redisplay)
			//       recognize "readonly" in readControls
			//echo '<br><br>'.__CLASS__."->".__FUNCTION__.':setting '.$this->name;
			//debug_print_backtrace();
			if ($_REQUEST[$this->name] != '') {
				$this->value = $_REQUEST[$this->name];
			} else if ($_REQUEST[$this->name.'_selection'][0] != $this->selectMessage) {
				$this->value = $_REQUEST[$this->name.'_selection'][0];
			} else {
				$this->value = '';
			}
		}
	}

	public function render($demo=FALSE,$index=1,$indent=0) {
		$prefix = str_pad("",1+$indent*3);
		
		ob_start();
		
		$ro = "";
   		if ($this->readonly) {
			$ro = ' readonly ';
		}
			
		$extra = $this->controlStyle().$this->extraHTML;
		
		//if ($this->size > 0) {
		//	$extra = sprintf(' style="width:%dpx; overflow:hidden" ', $this->size).$extra;
		//}
		//		$this->html = '<div class="'.get_class($this).'"><input   type="text" name="%s" value="%s" size="%d"'.
		//	' maxlength="%d" title="%s" %s><br><select name="%s_selection[]" 
		// size="%s" title="a non-blank value choosen here will be taken as input for the previous entry-field">';
		
		printf($this->html, $this->name, $this->value, $this->size, 
			$this->maxlength, $this->tiptext, $ro, $extra, $this->name, 
			/*$this->size*/1);

		echo '<option>'.$this->selectMessage."</option>
"; 

		foreach ($this->selections as $item) {
			echo '<option>'.$item."</option>
";
		}
		if ($this->mandatory) {
			$postfix = '*';
		} else {
			$postfix ='';
		}
		echo '</select></fieldset>';
		$html = ob_get_contents();
		ob_end_clean();
   		return $prefix.$html.$postfix;
	}
}

?>