<?PHP
/**
 * table browsing window
 *
 * This class serves as a base class for windows that allow to scroll through a set of
 * database records. It works with an *unlimited* number of records and supports
 * page-wise scrolling for such large sets. The window is also a template/demo and is
 * used in the GGFDemo application.
 *
 * assume model[0] = empty or id of a query object. If an id is given, the query must exist in the GGFQuery table in the DB
 * if model[0] is empty following settings are optional:
 * model["_where"] = a preset GGFSqlfilter object
 * model['_order'] = a preset GGFSqlSorter object
 * model['_columns'] = a preset array of column names
 * model['_columnsize'] = a preset array of column widths (in characters)
 * model['_query'] = a preset GGFQuery object (this will override all of the above)
 * model[1] = a preset GGFQuery object (this will override all of the above)
 * 
 * Note: all these settings can be changed interactively by the user.
 * 
 * additional optional settings:
 * model["_updateDialogClass"] = name of a GGFControlDialog class
 * model["_insertDialogClass"] = name of a GGFControlDialog class
 * model["_copyDialogClass"]   = name of a GGFControlDialog class
 * model["_updateAfterInsert"] = boolean
 * model["_etypeName"] = name of a GGFERModel claa
 * 
 * @todo comment, test
 * @package GGF
 * @version 4.2
 * @since 2.0
 * @author Gerald Zincke, Austria
 * @copyright 2005, 2012 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.html#top
 */
class GGFControlBrowserWindow extends GGFControlWindow {
	// this window is built with frames. we need at least one additional pane per frame

	/** @var Pane */
 	protected $headerPane;
	/** @var Pane */
	protected $scrollPane;
	/** @var Pane */
	protected $footerPane;
	/** @var Pane */
	protected $listPane;
	/** @var Pane */
	protected $scrollBar;
	/** @var Pane */
	protected $top;
	/** @var Pane */
	protected $up;
	/** @var Pane */
	protected $down;
	/** @var Pane */
	protected $bottom;
	protected $lineHeight = 20;
	protected $pageOffset = 110;
	protected $enableCopy;

	function __construct($contextID) {
       		parent::__construct($contextID);
       		//GGF3.0
       		$this->ETypeName = "user"; 	// initialize the name of the main model object class of the window
       		//GGF3.5
       		$this->enableCopy = true; 	// enables copy feature in browser window
   	}

   	/**
   	 * takes care for a standard window title
   	 */
	protected function openHead() {
		global $appname;
		if ($this->windowTitle>"") {
			echo "<title>".$this->windowTitle." </title>";
		} else {
			echo "<title>".$this->ETypeName." Browser - ".$appname." </title>";
		}
		if (isset($faviconURL)) {
			echo '<link rel="icon" type="image/x-icon" href="'.$faviconURL.'">';
		}
	}

	//---utility--------------------------

	/**
   	 * resets the query object in case filters, sorters or column selections are changed
   	 * and resets the direct access url in the system tray
   	 */
	protected function resetQuery() {
		unset($this->myModel["_query"]);
		$p = $this->pane->named("shortcut");
		if (!($p == 0)) {
			$p->setURL("--> invalid because query not saved <--");
		}
	}
   	/**
   	 * creates a comma seperated string from an array
   	 *
   	 * @param string[] containing the partial strings
   	 * @return string comma seperated string
   	 */
	public function commaString($a) {
		if (count($a) > 0 ) {
			$str = $a[0];
			for ($i=1; $i<(count($a)); $i++) {
				$str = $str.', '.$a[$i];
			}
		} else {
			$str = "";
		}
		return $str;
	}

	/**
	 * checks if a call for pageUp (by pressing the button) is possible
	 * 
	 * @return boolean true if pageUp could be called
	 */
	public function canPageUp() {
   		if($this->myModel["_pagesize"] > 0 ) { // LARGE list, scroll
			// now it depends if we are at page 1
			return (($this->myModel["_page"]) > 1);
		} else { // all data are displayed at once
			return FALSE;
		}
	}

	/**
	 * checks if a call for pageDown (by pressing the button) is possible
	 * 
	 * @return boolean true if pageDown could be called
	 */
	public function canPageDown() {
   		if($this->myModel["_pagesize"] > 0 ) { // LARGE list, scroll
			// now it depends if we are at the end 
			return ((count($this->myModel["_list"])) >= ($this->myModel['_pagesize']));
			// if the page is not full, its the end, other wise there may be additional data
			// but we don't know, so let the button be enabled
		} else { // all data are displayed at once
			return FALSE;
		}
	}

	//GGF3.2
	/**
	 * returns the sorter object of the window
	 * 
	 * @return GGFSqlSorter 
	 */
	public function order() {
		// give the sorter object
		if (!isset($this->myModel["_order"])) {
			$this->myModel['_order'] = new GGFSqlSorter();
		}
		return $this->myModel["_order"];
	}
	

	//---Model handling---------------------------------------------
	
	/**
   	 * creates the array of fields which are selected by the SQL statement
   	 *
   	 * the array is stored in $this->myModel["_columns"]
   	 *
   	 * additionally the array of all available columns is stored in $this->myModel["_allcolumns"]
   	 */
   	protected function createColumns() {
   		// the name of db columns to displayed
   		if (!isset($this->myModel["_columns"])) {
   		    	//$allcols =    array('userid',   'firstname','lastname','level',    'preferences','last_change');
   		    	//GGF 3.0
   		    	//echo "n:".$this->ETypeName;
   		    	//print_r($this->ERModel);
   		    	$et = $this->ERModel->entityTypeNamed($this->ETypeName);
   		    	$this->myModel["_columns"] = $et->defaultAttributeNames();
   		    	$this->myModel["_columnsize"] = array();
   		    	$this->myModel["_allcolumns"] = $et->attributeNames();  
   		}
	}

	/**
	 * creates the array of tables which the SQL statement selects from
	 *
	 * defines the tables used to select the data
	 *
	 * the array is stored in $this->myModel["_from"]
	 */
   	protected function createFrom() {
   		// the name of the table(s)
   		if (!isset($this->myModel["_from"])) {
   		    $type = $this->ERModel->entityTypeNamed($this->ETypeName);
   		    $this->myModel["_from"] = $type->tableNames();
   		}
   	}
   	//GGF3.3
   	public function createStandardQuery($name="") {
   		global $appname;
   		if ($name =="") {
   			$qname = $appname."-".$this->ETypeName;
   		} else {
   			$qname = $name;
   		}
   		$q = new GGFQuery($_SESSION["userid"],
   				$this->ETypeName,
   				get_class($this),
   				$qname,
   				$this->myModel['_where'],
   				$this->myModel['_order'],
   				$this->myModel['_columns'],
   				$this->myModel['_columnsize']
   		);
   		return $q;
   		//print_r($q);
   	}

   	/**
   	 * creates the where-clause of the SQL statement
   	 *
   	 * defines the rules which the SQL statement selects by
   	 *
   	 * uses a GGFSqlFiter object stored in $this->myModel['_where'] which creates the actual where-clause
   	 */
   	protected function createFilter() {
   		// an object that allows to construct a where-clause
   		
   		if (!isset($this->myModel["_where"])) {
   		    //$this->myModel['_where'] = new GGFERFilter($this->ERModel->entityTypeNamed($this->ETypeName));
   		    $et = $this->ERModel->entityTypeNamed($this->ETypeName);
   		    $this->myModel["_where"] = $et->defaultFilter();
   		}
   	}	
   	
   	/**
   	 * creates the order-clause of the SQL statement
   	 *
   	 * defines the rules which the selected data is sorted by
   	 *
   	 * uses a GGFSqlSorter object stored in $this->myModel['_order'] which creates the actual order-clause
   	 */
   	protected function createOrder() {
   		// the order columns 
   		if (!isset($this->myModel["_order"])) {
   		    //$this->myModel['_order'] = new GGFSqlSorter();
   		    $et = $this->ERModel->entityTypeNamed($this->ETypeName);
   		    $this->myModel["_order"] = $et->defaultSorter();
   		}
   	}	
   	
   	/**
   	 * sets the number of rows per page
   	 * 
   	 * createFrom, createFilter have to be executed before
   	 *
   	 * result is stored in $this->myModel['_pagesize']
   	 */
   	protected function createPagesize() {
   		global $db;
   		global $errorStack;
   		global $largeListThreshold;
   		global $smallListPageSize;
   		
   		//GGF3.4
   		
 		$this->myModel['_pagesize'] = $smallListPageSize; // This is the default value
   		
   		if ($_COOKIE[GGFPageHeight] > 0) { // current size of listarea is known
   			//print_r(round(($_COOKIE[GGFPageHeight] - 110) / 12));
   			$this->myModel['_pagesize'] = round(($_COOKIE[GGFPageHeight] - $this->pageOffset) / $this->lineHeight);
   		} // otherwise no Info from Javascript
   		
   		// now check, how many records it will be
   		// this is done every time because the db contents or where clause may change
   		$filter = $this->myModel['_where'];
		//GGF3.0
		$this->myModel['_list_count'] = $filter->count(); 
		if ($this->myModel['_list_count'] < $largeListThreshold) {
			$this->myModel['_pagesize'] = 0;
		}
   	}	
   	
   	/**
   	 * creates the page to display
   	 *
   	 * stored in $this->myModel["_page"]
   	 */
   	protected function createPage() {
   		// the page to display
   		global $traceLevel;
   		if (!isset($this->myModel["_page"])) {
   			$this->myModel["_page"] = 1;
   		} else {
   			//echo "was not reset";
   		}
   		if ($traceLevel>=2) { 
			error_log_adv ("$this->appname ".__CLASS__."->".__FUNCTION__.': $this->myModel["_page"]='.
			$this->myModel["_page"]);
		}		
   	}	
   	
   	/**
   	 * creates the whole SQL select statement used to read the data for the list
   	 *
   	 * @return string SQL statement
   	 */
   	protected function createSELECT($noLimit=FALSE) {
   		global $db;
   		$filter = $this->myModel['_where'];
   		$sorter = $this->myModel['_order'];
   		
   		if ($_COOKIE[GGFPageHeight] > 0) { // current size of listarea is known
   			//print_r(round(($_COOKIE[GGFPageHeight] - 110) / 12));
   			$this->myModel['_pagesize'] = round(($_COOKIE[GGFPageHeight] - $this->pageOffset) / $this->lineHeight);
   		} // otherwise no Info from Javascript
   		 
   		$pageSize = $this->myModel["_pagesize"];
   		if ($noLimit) {
   			$pageSize = 0;
   		}
   		
   		//GGF3.0
   		$et = $this->ERModel->entityTypeNamed($this->ETypeName);
			
   		$this->myModel['_sql'] = $et->createSelect(
   			$this->myModel['_columns'],
   			$filter,
   			$sorter, 
   			$this->myModel["_page"],
   			$pageSize);
   				
		return $this->myModel['_sql'];
	}

	//GGF3.4
	protected function initModel($mc) {
   		// prepare the model object
   		global $traceLevel;
   		global $db;
		global $errorStack;
		global $appname;
   			
		//print_r($mc->model);
		if (isset($mc->model["_etypeName"])) {
   			$this->ETypeName = $mc->model["_etypeName"];
   		}
		//GGF3.3
		if ($mc->model ==0) {
   			$mc->model = array();
   		} else {
   			//print_r($mc->model);
   			if (isset($mc->model["_query"])) { // query object here
	   			$query = $mc->model["_query"];
   	   		} else { 
	   	   		if (isset($mc->model[1])) { // query object here
	   				$query = $mc->model[1];
		   		}
	   			if (isset($mc->model[0])) { // query-id here
		   			
				   	$this->myModel = $mc->model;
					//error_log_adv('--->'.__File__.__Line__.print_r($mc->model[1],TRUE));
					// the database record has not yet been read for this context
					$res = $db->execSQL('SELECT * FROM GGFQuery WHERE ID='.$mc->model[0]);
			
					if (!$db->OK()) {
						$errorStack->pushMsg("GGFControlBrowserWindow: Cannot read query definition '".$mc->model[0]."' .");
						// not necessary to display error here, done in open
						return;
					} else {
						$row = $db->fetchRow($res);
						unset($mc->model[0]);
		   			
			   			$query = new GGFQuery();
			   			$query->setRow($row);
			   		}
			   	}
		   	}
			
		   	
		   	if (!$query=="") {
	 			//print_r($row);
	   			//echo "-------------";
				//print_r($query);
	   			//echo "-------------";
				
   				$mc->model["_query"] = $query;
   				$mc->model["_queryname"] = $query->qname(); // needed separately for saveQuery()
   				
			    $this->ETypeName = $query->etypename();
   				$mc->model["_etypeName"] = $query->etypename();
   				
   				
   				//print_r($this->ETypeName);
   				$mc->model["_where"] = $query->filter();
   				$mc->model['_order'] = $query->sorter();
   				$mc->model['_columns'] = $query->columns();
   				$mc->model['_columnsize'] = $query->columnSize();
   				$et = $this->ERModel->entityTypeNamed($this->ETypeName);
   				if ($et == 0) {
   					$errorStack->pushMsg(__Class__.": Cannot open Query ".$query->qname());
				} else {
   		  			$mc->model["_allcolumns"] = $et->attributeNames();
   		    		}
				
				
	   		}
   		}

   		$et = $this->ERModel->entityTypeNamed($this->ETypeName);
   		if ($et == 0) {
   					$errorStack->pushMsg(__Class__.": Model ".$this->ETypeName." is not (globally) defined. Check class ".$appname."Model");
					$errorStack->pushMsg(__Class__.": Cannot open Query ".$query->qname());
					$this->eventDisplayError($returnCallback="eventClose");
					$mc->save();
					exit;
   		}
   				
   		$this->myModel = $mc->model; 
   		
   		//print_r($this->myModel);
   		
   		//GGF3.3 - seems the workaround not needed anymore
   		//$this->myModel = $_SESSION["_amod".$this->myContextID]; //this is a workaround, because session does not save model in context correctly
   		
		if ($traceLevel>=2) {
			error_log_adv("initializing Model for context:".$this->myContextID.' session:'); 
			//error_log_adv (print_r($_SESSION,TRUE));
			if ($traceLevel>=3) {
				error_log_adv ("$this->appname ".__CLASS__."->".__FUNCTION__.": dump myModel[]: _page   =".$this->myModel["_page"]." _columns=".$this->myModel['_columns']." _order  =".print_r($this->myModel['_order'], TRUE));
			}
		}
			
		
		$this->createColumns();	
		$this->createFrom();	
		$this->createFilter();	
		$this->createOrder();	
		$this->createPagesize();	
		$this->createPage();	
		$this->createSELECT();
   		
   		
   		if (isset($this->myModel["_list"]) AND (!$this->myModel["_list"] == 0)) {
			// do not read every time from db, because changes via subdialogs (see eventSelectLevel) would not work
		} else {
			$this->refreshModel($mc);		
		}
		if ($traceLevel>=2) { 
			error_log_adv ("$this->appname ".__CLASS__."->".__FUNCTION__.": dump myModel[]: _page   =".$this->myModel["_page"]." _columns=".$this->myModel['_columns']." _order  =".print_r($this->myModel['_order'],TRUE));
		}
		//GGF3.2.2
		if (!isset($this->myModel["_updateDialogClass"])) {
			$this->myModel["_updateDialogClass"] = "GGFControlUpdateDialog";
		}
		if (!isset($this->myModel["_insertDialogClass"])) {
			$this->myModel["_insertDialogClass"] = "GGFControlInsertDialog";
		}
		if (!isset($this->myModel["_copyDialogClass"])) {
			$this->myModel["_copyDialogClass"] = $this->myModel["_insertDialogClass"];
		}
		if (!isset($this->myModel["_updateAfterInsert"])) {
			$this->myModel["_updateAfterInsert"] = true;
		}
		$mc->model = $this->myModel;
		$mc->save();
	}

	protected function refreshModel($mc) {
		global $db;
		global $errorStack;
		global $traceLevel;
		// the database record has not yet been read for this context
		$res = $db->execSQL($this->createSELECT());
	
		if (!$db->OK()) {
			$errorStack->pushMsg(__File__."->refreshModel: Cannot read table.");
			// not necessary to display error here, done in open
		} else {
			$this->myModel["_list"] = array();
			while($row = $db->fetchRow($res)) {
				array_push($this->myModel["_list"], $row);
			}
			if ($traceLevel>=2) { 
				error_log_adv ("$this->appname ".__CLASS__."->".__FUNCTION__.": ".
				count($this->myModel["_list"]).' records written to $this->myModel["_list"]');
			}
			$mc->model = $this->myModel;
			$mc->save();
		}
	}

	protected function initWindow($mc) {
		// add controls there
		/*
		
		The structure of this window is rather complex. It consists of several frames
		.------------------------------------------------------------------------------.
		| main frameset = $this->pane (containing frames 1,2 and 3)                    |
		|.----------------------------------------------------------------------------.| 
		|| frame1 = $this->headerPane                                                 ||
		|| (here the menu and the prompt are displayed)                               ||
		|:----------------------------------------------------------------------------:|
		|| frame2, frameset = $this->scrollPane                                       ||
		||.---------------------------------------------.----------------------------.||
		||| frame4 = $this->listArea                    | frame5 = $this->scrollBar  ||| 
		||| (here the rows of the tables are displayed) | (frameset for the panes:)  |||
		|||                                             |.--------------------------:|||
		|||                                             || topPane                  ||||
		|||                                             |:--------------------------:|||
		|||                                             || upPane                   ||||
		|||                                             |:--------------------------:|||
		|||                                             || downPane                 ||||
		|||                                             |:--------------------------:|||
		|||                                             || bottomPane               ||||
		|||                                             |'--------------------------'|||
		||'---------------------------------------------'----------------------------'||
		|'----------------------------------------------------------------------------'|
		|| frame3 = $this->footerPane                                                 ||
		|| (display of window status)                                                 ||
		|'----------------------------------------------------------------------------'|
		'------------------------------------------------------------------------------'
		
		See also function openBody.
		*/
		
		//for direct echo		
		$this->pane = new $this->myContainerClass($this,"_main");
		$this->pane->add(new GGFGenericControl("mainfs", '
			<frameset rows="70,*,25">  <!-- Frameset-Definition -->
			  <frame src="'.$mc->frameCallbackURL(1).'" name="headerPane" marginwidth="0" marginheight="0" frameborder="0" framespacing="0" border="0" scrolling="no" >
			  <frame src="'.$mc->frameCallbackURL(2).'" name="scrollPane" marginwidth="0" marginheight="0"  frameborder="0" framespacing="0" border="0">
			  <frame src="'.$mc->frameCallbackURL(3).'" name="footerPane" bgcolor="#E0E0E0" marginwidth="0" marginheight="0"  frameborder="0" framespacing="0" border="0">
			  <noframes>
			    Your Web-Browser cannot show frames. Use a modern browser.
			  </noframes>
			</frameset>'));

		//frame1
		$this->headerPane = new GGFControlPane($this,"_headerpane");
		$menu = new GGFMenu($this,"_menu");
		$menu->add(new GGFMenuItem("Insert",  "eventInsert", 		"isLoggedOn",	"insert a new row"));
		$menu->add(new GGFMenuItem("Columns", "eventSelectColumns", 	"",		"select the columns to display"));
		$menu->add(new GGFMenuItem("Sort",    "eventUpdateSorter", 	"",		"specify sort criteria"));
		$menu->add(new GGFMenuItem("Filter",  "eventUpdateFilter", 	"",		"specify criteria to select rows"));
		$menu->add(new GGFMenuItem("Queries", "eventSelectQuery", 	"",     	"select a query"));
		$menu->add(new GGFMenuItem("Save Query", "eventSaveQuery", 	"isLoggedOn",	"save current view as a query"));
		//$menu->add(new GGFMenuItem("Options", "eventOptions", 		"isLoggedOn",	"set options"));
		$menu->add(new GGFMenuItem("Close",   "eventClose", 		"",		"close window"));
		$this->headerPane->add($menu);
		$main = new GGFControlPane($this,"_main");
		$main->newLine();
		$main->newSpace();
		$main->add(new GGFStaticfield("prompt", "Browse table, select and click an action."));
		$this->headerPane->add($main);
		$this->pane->addFrame(1,$this->headerPane);
		
		//frame2
		$this->scrollPane = new GGFControlPane($this,"_scrollPane");
		$this->scrollPane->setIsFrameset();
		
		$this->pane->addFrame(2,$this->scrollPane);

		//frame 3
		$this->footerPane = new GGFControlPane($this,"_footerPane");
		//$this->footerPane->setStyle('font-family:Helvetica,Helv; font-size:8pt;');  
		$this->footerPane->newRow(new GGFStaticfield("status", $this->windowStatus(),0,"displays the current status of the window"),"text-align:left;  ","width:100%; font-family:Arial; font-size:0.8em;","width:98%");
		$this->footerPane->newSpace(3);
		//$this->footerPane->add(new GGFStaticHyperlinkfield("browseStatus", $this->callbackURL("eventUpdateFilter"), "unfiltered", 0, $tiptext="see the filter definition; if any",'target="_top"'));
		$this->footerPane->add(new GGFPseudoPushbutton("browseStatus", "unfiltered", "eventUpdateFilter","","",$tiptext="see the filter definition; if any",$extraHTML="", $target="_top"));
		$this->footerPane->newCell($this->systemTray($mc),"text-align:right;");
		$this->footerPane->setBodyExtraHTML('background-color:#E0E0E0; font-family:Arial; font-size:0.8em; border-width:2px; border-color:#E0E0E0; border-style:inset; padding:0px;');
		$this->pane->addFrame(3,$this->footerPane);
		
		//frame 4
		$this->listPane = new GGFControlPane($this,"_listArea");
		//GGF3.5
		if ($this->enableCopy) {
			$copyCb = "eventCopy";	// copy feature enabled
		} else {
			$copyCb = "";			// copy feature disabled
		}
		$this->listPane->add(new GGFListArea($name="_list", 
			$columns = array(), 
			$rows=array(),  
			$updateCallback="eventUpdate", 
			$deleteCallback="eventDelete",
			//GGF3.2
			/*$copyCallback="",*/ 
			//GGF3.5
			$copyCallback=$copyCb,
			$sortCallback="eventSortColumn",
			$pageSize=0, $tiptext="click to view and update", $extraHTML=""));
		$this->pane->addFrame(4,$this->listPane);
	
		//frame5
		$this->scrollBar = new GGFControlPane($this,"_scrollbar");
		// the scrollbar may be empty or not, depending on the list size
		// see openBody()
        	$this->scrollBar->setIsFrameset();

		$this->pane->addFrame(5,$this->scrollBar);
			
		//frame6-9	
		$this->top = new GGFFormPane($this,"_toppane");
		$this->top->add(new GGFImagePushButton($name="_top", $value="top", 
			$image="GGFIcons/GGFtop.jpg", $callback="eventTop", $validator="canPageup", 
			$size=0, $tiptext="top"));
		$this->top->setBodyExtraHTML('margin:0px; background-color:#E0E0E0; font-family:Arial; border-width:4px; border-color:#E0E0E0; border-style:outset; padding:0px;');
		$this->pane->addFrame(6,$this->top);

		$this->up = new GGFFormPane($this,"_uppane");
		$this->up->add(new GGFImagePushButton($name="_up", $value="up", 
			$image="GGFIcons/GGFup.jpg", $callback="eventUp", $validator="canPageup", 
			$size=0, $tiptext="up"));
		$this->up->setBodyExtraHTML('margin:0px; background-color:#E0E0E0; font-family:Arial; border-width:4px; border-color:#E0E0E0; border-style:outset; padding:0px;');
		$this->pane->addFrame(7,$this->up);

		$this->down = new GGFFormPane($this,"_downpane");
		$this->down->add(new GGFImagePushButton($name="_down", $value="down", 
			$image="GGFIcons/GGFdown.jpg", $callback="eventDown", $validator="canPageDown", 
			$size=0, $tiptext="down"));
		$this->down->setBodyExtraHTML('margin:0px; background-color:#E0E0E0; font-family:Arial; border-width:4px; border-color:#E0E0E0; border-style:outset; padding:0px;');
		$this->pane->addFrame(8,$this->down);

		$this->bottom = new GGFFormPane($this,"_bottompane");
		$this->bottom->add(new GGFImagePushButton($name="_bottom", $value="bottom", 
			$image="GGFIcons/GGFbottom.jpg", $callback="eventBottom", $validator="canPageDown", 
			$size=0, $tiptext="bottom"));
		$this->bottom->setBodyExtraHTML('margin:0px; background-color:#E0E0E0; font-family:Arial; border-width:4px; border-color:#E0E0E0; border-style:outset; padding:0px;');
		$this->pane->addFrame(9,$this->bottom);

		$pane = $this->pane->named("_customSystemTray");
		
		$pane->add(new GGFMenuItem('<img src="GGFIcons/GGFfirst.gif" width="16" height="16" border="0">', 'eventTop', 'canPageup', 'click to view the first page'));
		$pane->add(new GGFMenuItem('<img src="GGFIcons/GGFprev.gif" width="16" height="16" border="0">', 'eventUp', 'canPageup', 'click to view the previous page'));
		$pane->add(new GGFMenuItem('<img src="GGFIcons/GGFnext.gif" width="16" height="16" border="0">', 'eventDown', 'canPageDown', 'click to view the next page'));
		$pane->add(new GGFMenuItem('<img src="GGFIcons/GGFlast.gif" width="16" height="16" border="0">', 'eventBottom', 'canPageDown', 'click to view the last page'));
		$pane->add(new GGFMenuItem('<img src="GGFIcons/GGFcsv.gif" width="16" height="16" border="0">',
					"eventExport",
					"",
					"Export the model data to csv format",
		                        $target="_blank"
					));
		

		$mc->model = $this->myModel;
		$mc->save();
	}		

	/**
	* returns the primary key value of a given row
	* 
	* @param array $row
	* @return integer the primary key value of current data set
	*/
	public function PKValue ($row) {
		//GGF3.1.1
		$et = $this->ERModel->entityTypeNamed($this->ETypeName);
		$pk = $et->primaryKey();
		return $row[$pk[0]];
	}
	
	public function directAccessURL() {
		//create an URL to the dispatcher
		// if the window was opened by a query and the query is still unchanged...
		//print_r($this->myModel["_query"]);
		if (isset($this->myModel["_query"])) {
			$r = $this->myModel["_query"]->row();
			if ($r["id"] > 0) {
				$r = $this->myModel["_query"]->row();
				$c = &windowContext($this->myContextID, "");
				return $c->directAccessURL("pkvalue=".$r["id"]);
				// note: this url corresponds to the context-generation code in the constructor
			}
		} 
		return "--> invalid because query not saved <--";		
	}
	
	//---event handling---------------------------------------	

	/**
	* creates a CSV representation of the data table
	* 
	*/
	protected function eventExport() {
		$mc = &windowContext($this->myContextID, "");
		global $db;
		global $errorStack;
		global $traceLevel;
		// the database record has not yet been read for this context
		$res = $db->execSQL($this->createSELECT(TRUE));
	
		if (!$db->OK()) {
			$errorStack->pushMsg(__File__.": Cannot read table.");
			// not necessary to display error here, done in open
		} else {
			$colarr = $this->myModel["_columns"];
			foreach ($colarr as $col) {
				echo $col."\t";
			}
			echo "\r\n";

			while($row = $db->fetchRow($res)) {
				//GGF3.0
				for ($i = 0; $i < count($colarr); $i++) {
					echo $row[$colarr[$i]]."\t";
				}
				echo "\r\n";

			}
		}
		//GGF3.0
		exit;
		
	}
	
	/**
	* select the columns to be displayed
	* 
	*/
	protected function eventSelectColumns() {	
		$myContext = &windowContext($this->myContextID,0);

		//print_r($this->myModel["_columnsize"]);
		$myContext->openDialogOn("GGFControlColumnsDialog", 
		array($this->myModel["_columns"],$this->myModel["_allcolumns"],"Select Columns to display",$this->myModel["_columnsize"]), 
		"eventSelectColumnsReturn");
		exit;
	}

	protected function eventSelectColumnsReturn($subContextID) {
		// process selected level
   		$subContext = &windowContext($subContextID,0);
   		if ($subContext->result) { // OK, sort criteria selected
   			$this->myModel["_columns"] = $subContext->model[0];
   			$this->myModel["_columnsize"] = $subContext->model[3];
   			//print_r($this->myModel["_columnsize"]);
   			// print_r($this->myModel);
			// error_log_adv('SortDialog returned subcontext:'.print_r($subcontext, true));
   		
		   	$mc = &windowContext($this->myContextID,0);
			$this->resetQuery();
		
			$mc->model = $this->myModel;
			$mc->save();
			// print_r($this->myModel);
			$this->createSELECT();
			$this->refreshModel($mc);
		}
	}

	/**
	* specify the sort criteria
	* 
	*/
	protected function eventUpdateSorter() {	
		$myContext = &windowContext($this->myContextID,0);
		$myContext->openDialogOn("GGFControlSortDialog", 
		array($this->myModel["_order"],$this->myModel["_allcolumns"],$this->ETypeName), 
		"eventUpdateSorterReturn");
		exit;
	}

	protected function eventUpdateSorterReturn($subContextID) {
		// process selected level
   		$subContext = &windowContext($subContextID,0);
   		if ($subContext->result) { // OK, sort criteria selected
   			$this->myModel["_order"] = $subContext->model[0];
   			// print_r($this->myModel);
			// error_log_adv('SortDialog returned subcontext:'.print_r($subcontext, true));
   		
		   	$mc = &windowContext($this->myContextID,0);
			$this->resetQuery();
		
			$mc->model = $this->myModel;
			$mc->save();
			// print_r($this->myModel);
			$this->createSELECT();
			$this->refreshModel($mc);

		}
	}

	/**
	* specify the filter for the rows to be displayed
	* 
	*/
	protected function eventUpdateFilter() {	
		$myContext = &windowContext($this->myContextID,0);
		
		$myContext->openDialogOn("GGFERFilterDialog", 
		array(clone($this->myModel["_where"]),array()),
		"eventUpdateFilterReturn");
		exit;
	}

	protected function eventUpdateFilterReturn($subContextID) {
		// process selected level
   		$subContext = &windowContext($subContextID,0);
   		if ($subContext->result) { // OK, filter criteria updated
   			$this->myModel["_where"] = $subContext->model[0];
   			// print_r($this->myModel);
			// error_log_adv('SortDialog returned subcontext:'.print_r($subcontext, true));
			
			unset($this->myModel['_pagesize']);
			unset($this->myModel['_page']);
			
			$this->createPagesize();
			$this->createPage();
   		
		   	$mc = &windowContext($this->myContextID,0);
			$this->resetQuery();
		
			$mc->model = $this->myModel;
			$mc->save();
			// print_r($this->myModel);
			$this->createSELECT();
			$this->refreshModel($mc);

		}
	}

	/**
	* specify the query for the rows to be displayed
	* 
	*/
	protected function eventSelectQuery() {	
		$myContext = &windowContext($this->myContextID,0);
		
		$q = $this->createStandardQuery();
		
		if (isset($this->myModel["_query"])) {
			$q->setQname= $this->myModel["_query"]->qname();
		}
		
		$myContext->openDialogOn("GGFQuerySelectDialog", 
		array($q,$q->qname()),
		"eventSelectQueryReturn");
		exit;
	}

	protected function eventSelectQueryReturn($subContextID) {
		// process selected level
   		$subContext = &windowContext($subContextID,0);
   		if ($subContext->result) { // OK, new query selected
   			$query = $subContext->model[0];
   			$this->myModel["_where"] = $query->filter();
   			$this->myModel['_order'] = $query->sorter();
   			$this->myModel['_columns'] = $query->columns();
   			$this->myModel['_columnsize'] = $query->columnSize();
			$this->myModel["_query"] = $query;
			$this->myModel["_queryname"] = $query->qname(); // needed separately for saveQuery()
			
   			// print_r($this->myModel);
			// error_log_adv('SortDialog returned subcontext:'.print_r($subcontext, true));
			
			unset($this->myModel['_pagesize']);
			unset($this->myModel['_page']);
			
			$this->createPagesize();
			$this->createPage();
   		
		   	$mc = &windowContext($this->myContextID,0);
			$mc->model = $this->myModel;
			$mc->save();
			// print_r($this->myModel);
			$this->createSELECT();
			$this->refreshModel($mc);

		}
	}

	/**
	* save current view as query
	* 
	*/
	protected function eventSaveQuery() {	
		$myContext = &windowContext($this->myContextID,0);
		
		$q = $this->createStandardQuery($this->myModel["_queryname"]);
		
		//if (isset($this->myModel["_queryname"])) {
		//	$q->setQname= $this->myModel["_queryname"];
		//}
		
		$myContext->openDialogOn("GGFQuerySaveDialog", 
		array($q,$q->qname()),
		"eventSaveQueryReturn");
		exit;
	}
	protected function eventSaveQueryReturn($subContextID) {
   		$subContext = &windowContext($subContextID,0);
   		if ($subContext->result) {
   			$q = $subContext->model[0];
			$this->myModel["_query"] = $q;
   			$this->myModel["_queryname"] = $q->qname();
   	   	}
		$mc = &windowContext($this->myContextID,0);
		$mc->model = $this->myModel;
		$mc->save();
	}
	
	/**
	 * selects all pk values and creates an info text, which is displayed in eventDelete
	 *
	 * @return unknown
	 */
	protected function getEntityInfo() {
		/*global $db;
		global $errorStack;
		$info = "";
		
		$et = $this->ERModel->entityTypeNamed($this->ETypeName);
		$pk = $et->primaryKey();
		$filter = new GGFERFilter($et);
		
		$key = split(',', $this->myModel['_selection']);
		$idx = 0;
		foreach($pk as $pkName) {
			$filter->add(new GGFSqlFilterClause("AND", $pkName, "=", $key[$idx]));
			$idx++;
		}
		$res = $db->execSQL($et->createSelect(array(), $filter, new GGFSqlSorter()));	// selects only pk values
		
		if (!$db->OK()) {
				$errorStack->pushMsg("GGFControlBrowserWindow - eventDeleteReturn: Cannot read record.");
		} else {
			$row = $db->fetchRow($res);
			$comma = "";
			foreach($pk as $pkName) {
				$info = $info.$comma.$row[$pkName];
				$comma = ", ";
			}
		}*/
		
		return $this->myModel["_selection"];
	}

	/**
	* delete a row
	* 
	*/
	protected function eventDelete() {
		// delete record
		// this is called by something like: 
		// http://localhost/ggf/GGFDispatch.php?classname=GGFBrowserWindow
		// &contextID=5&PHPSESSID=3b384237f08cc95ab542acff18bde3a2
		// &callback=eventDelete&pkvalue=last
		
		// remember row to be deleted
		$this->myModel["_selection"] = $_REQUEST["pkvalue"];
		$this->saveModel();			

		if ($this->donotDelete()) {
			// do nothing
		} else {
			$mc = &windowContext($this->myContextID,0);
			$mc->model = $this->myModel;
			$mc->save();
	
			$info = $this->getEntityInfo();
			
			messageBox($this->myContextID,
				"Are you sure that you want to delete record: "./*$this->myModel["_selection"].*/$info." ?",
				2,
				"eventDeleteReturn");
			// results of dialog will be processed in eventDeleteReturn
			exit;
		}
   	}

	//GGF3.4
	protected function donotDelete() {
		global $errorStack;
		$donotDel = false;
		
		/*if(!$this->isLoggedOn()) {
			messageBox($this->myContextID,"You cannot delete records, because you're not logged on",
				1,
				"");
			exit;
		}
		if ($this->myModel["_selection"]=="admin") {
			messageBox($this->myContextID,"You cannot delete user 'admin'",
				1,
				"");
			exit;
		}*/
		if(!$this->isLoggedOn()) {
			$errorStack->pushUsrMsg(1,"You cannot delete records, because you're not logged on.");
			$donotDel = true;
		}
		if ($this->myModel["_selection"]=="admin") {
			$errorStack->pushUsrMsg(1,"You cannot delete user 'admin'.");
			$donotDel = true;
		}
		
		return $donotDel;
		
	}
	
	/** supports composite keys */ 
	protected function createDELETE() {
		global $db;
		$et = $this->ERModel->entityTypeNamed($this->ETypeName);
		$tn = $et->tableNames();
		$pk = $et->primaryKey();
		//$sql = "DELETE FROM ".$tn[0]." WHERE ".$pk[0]." = '".$this->myModel["_selection"]."'";
		
		$key = split(',', $this->myModel['_selection']);
		$sql = "DELETE FROM ".$tn[0]." WHERE ";
		
		$idx = 0;
		$and = "";
		foreach($pk as $pkName) {
			$sql = $sql.$and.$pkName."= '".$db->escape_string($key[$idx])."'";
			$and = " AND ";
			$idx++;
		}
		
		return $sql;
	}
	
	protected function eventDeleteReturn($subContextID) {
   		$subContext = &windowContext($subContextID,0);
   		if ($subContext->result) {
   			global $db;
   			
			//GGF3.2.2
   			// do the delete action
   			$et = $this->ERModel->entityTypeNamed($this->ETypeName);
   			$tn = $et->tableNames();
			$db->execSQL($this->createDELETE() ,$tn[0]);
			// display of error done by framework
	   	}
		$mc = &windowContext($this->myContextID,0);
		$this->refreshModel($mc);
	}

	/**
	* update a row
	*/
   	protected function eventUpdate() {
		// update record
		
		// remember row to be updated
		$this->myModel["_selection"] = $_REQUEST["pkvalue"];
		$mc = &windowContext($this->myContextID,0);
		$mc->model = $this->myModel;
		$mc->save();
		// prepare the model for the update dialog = key of row, and placeholder for row
		$mod = array($this->myModel["_selection"],array());
			
		$mc->openDialogOn($this->myModel["_updateDialogClass"],$mod , $cont="eventUpdateReturn"  );
		exit;
   	}
   	
   	protected function eventUpdateReturn($subContextID) {
   		$subContext = &windowContext($subContextID,0);
   		
   		//if ($subContext->result) {
   			$mc = &windowContext($this->myContextID,0);
   			$this->refreshModel($mc);
   		//}
	}

	/**
	* insert a new row and update it
	* 
	*/
	protected function eventInsert() {
		// insert record
		$mc = &windowContext($this->myContextID,0);
		// error_log_adv('GGFBrowserWindow->eventInsert:'.print_r($myContext, true));
   		$mod = array(0,array());
		//GGF3.2.2		
		$mc->openDialogOn($this->myModel["_insertDialogClass"],$mod , $cont="eventInsertContinue"  );
		exit;
   	}
	protected function eventInsertContinue($subContextID) {
		// after insert, update new empty record
		$subContext = &windowContext($subContextID,0);
		if ($subContext->result == TRUE) {
			// user closed insert dialog with OK 
			$mod = $subContext->model;
			$mc = &windowContext($this->myContextID,0);
			$this->refreshModel($mc);
			
			if ($this->myModel['_updateAfterInsert']) {
				//GGF3.2.2
				$mc->openDialogOn($this->myModel["_updateDialogClass"],$mod , $cont="eventUpdateReturn"  );
				exit;
			}
		}
   	}

	/**
	* sort a column
	* 
	*/
	protected function eventSortColumn() {
		// sort by selected column

		// column to be sorted
		$col = $_REQUEST["col"];
		
		// look if sort column already a sortcriteria
		$sorter = $this->order() ;
		$criteria = $sorter->getCriteria();
		$i = 0;
		$done = False;
		//print_r ($col);
		//print_r ($criteria);
		//error_log_adv('searching for:'.$col);
				
			
		foreach ($criteria as $crit) {
			$chunk = explode(" ", $crit);
			//print_r ($chunk);
			//error_log_adv(print_r($chunk,TRUE));
			
			if (strcmp($chunk[0],$col)==0) { // this is already a sort column. just change the sort direction
				if (isset($chunk[1])) { // DESC
					$criteria[$i] = $col;	// defaults to ASC
				} else {
					$criteria[$i] = $col.' DESC';	// was ASC, now DESC
				}
				$sorter->setCriteria($criteria);
				$done = True;
				break;
			}
			$i = $i+1;
			
		}
		
		if (!$done) {
			$sorter= new GGFSqlSorter();
			$sorter->setCriteria(array($col));
		}

		$this->myModel["_order"] = $sorter;
   		
		$mc = &windowContext($this->myContextID,0);
		$this->resetQuery();
		$mc->model = $this->myModel;
		$mc->save();
		// print_r($this->myModel);
		$this->createSELECT();
		$this->refreshModel($mc);

	}

	/**
	* update and save options
	*
	* this is a placeholder for future use
	* @abstract
	*/
	protected function eventOptions() {
		// tbi
	}
	
	/**
	* copies a row
	* NOTE: if there are columns that have to be unique this function has to be overwritten
	* 
	*/
	protected function eventCopy() {
		global $db;
		global $errorStack;
		
		// select all columns from current record and write the to $this->myModel["_copy"]
		$this->myModel["_selection"] = $_REQUEST["pkvalue"];
		$et = $this->myModel["_where"]->type();
		if (!($et->rootTypeName()=="")) { // expanded entity type
			$et = $this->ERModel->entityTypeNamed($et->rootTypeName());
		}
		$tn = $et->tableNames();	// TODO: replace through string search/cut of ETypeName
		$pk = $et->primaryKey();
		//$columns = array_diff($this->myModel["_allcolumns"], $pk);
		$columns = array_diff($et->attributeNames(), $pk); // should work also if expanded entitytype is browsed
		
		$sql = "SELECT ".implode(',', $columns)." FROM ".$tn[0]." WHERE ";
	
		$key = split(',', $this->myModel["_selection"]);
		$and = "";
		$idx = 0;
		foreach ($pk as $pkName) {
			$sql = $sql.$and.$pkName."=".$key[$idx];
			$idx++;
			$and = " AND ";
		}
		
		$res = $db->execSQL($sql);
		if (!$db->OK()) {
			$errorStack->pushMsg("GGFControlBrowserWindow: Could not read values from record <b>".$this->myModel["_selection"]."</b> for copy.");
		} else {
			$row = $db->fetchRow($res);
			$this->myModel["_copy"] = $row;
			
			// calling copyDialogClass with values to copy in myModel[2]
			$mc = &windowContext($this->myContextID,0);
	   		$mod = array(0,array(), $this->myModel["_copy"]);	
			$mc->openDialogOn($this->myModel["_copyDialogClass"],$mod , $cont="eventCopyReturn");
			// exit
		}
	}
	
	protected function eventCopyReturn($subContextID) {

		$mc = &windowContext($this->myContextID,0);
		$this->refreshModel($mc);
	}
	
 	
	/**
	* scroll to the first page
	*
	* if the number of rows to be displayed is less than a threshold
	* records are displayed page per page
	* 
	*/
	protected function eventTop() {
		// first page
		if ($this->myModel["_page"]>1) {
			$mc = &windowContext($this->myContextID,0);
			$this->myModel["_page"] = 1;
			$this->refreshModel($mc);
			$mc->model = $this->myModel;
			$mc->save();
		}
	}
	
	/**
	* scroll to the last page
	*
	* if the number of rows to be displayed is less than a threshold
	* records are displayed page per page
	* 
	*/
	protected function eventBottom() {
		// scroll list to bottom of table
		global $smallListPageSize;
   		
		$mc = &windowContext($this->myContextID,0);
		//GGF3.2
		//$this->myModel["_page"] = 1 + floor($this->myModel['_list_count'] / ($this->myModel['_pagesize'] + 1) ) ;
		$this->myModel["_page"] = 1 + floor(($this->myModel['_list_count'] - 1) / $this->myModel['_pagesize'] );
		$this->refreshModel($mc);
		$mc->model = $this->myModel;
		$mc->save();
	}
	
	/**
	* scroll to the next page
	*
	* if the number of rows to be displayed is less than a threshold
	* records are displayed page per page
	* 
	*/
	protected function eventDown() {
		// next page
		global $smallListPageSize;
		//GGF3.2
		//if ($this->myModel["_page"] < (1 + floor($this->myModel['_list_count'] / ($this->myModel['_pagesize'] + 1) ))) {
		if ($this->myModel["_page"] < (1 + floor(($this->myModel['_list_count'] - 1) / $this->myModel['_pagesize']))) {
			$mc = &windowContext($this->myContextID,0);
			$this->myModel["_page"]++;
			$this->refreshModel($mc);
			$mc->model = $this->myModel;
			$mc->save();
		}
	}

	/**
	* scroll to the previous page
	*
	* if the number of rows to be displayed is less than a threshold
	* records are displayed page per page
	* 
	*/
	protected function eventUp() {
		// previous page
		if ($this->myModel["_page"]>1) {
			$mc = &windowContext($this->myContextID,0);
			$this->myModel["_page"]--;
			$this->refreshModel($mc);
			$mc->model = $this->myModel;
			$mc->save();
		}
	}

	
	
	//---Window Display---------------------------------------------------------
	
	protected function fillControls($mc) {
		// read hints in the baseclass when modifying this method
		if (isset($this->myModel["_query"])) {
			$this->pane->named("prompt")->setValue("Query: ".$this->myModel["_queryname"]);
			if ($this->shortCutEnabled) {
				$this->pane->named("shortcut")->setURL($this->directAccessURL());
			}
		}	    
		
		$listArea = $this->listPane->named("_list");
		$listArea->setValue($this->myModel["_list"]);
		
		$listArea->setColumns($this->myModel["_columns"]);
		//GGF3.5
		$listArea->setColumnSize($this->myModel["_columnsize"]);
		$et = $this->ERModel->entityTypeNamed($this->ETypeName);
		$en = array();
		
		//print_r($this->myModel["_columns"]);
		
		foreach($this->myModel["_columns"] as $colname) {
			$att = $et->getAttributeNamed($colname);
			//print_r($et);
			
			array_push($en,$att->externalName());
		}
		$listArea->setExternalColumnNames($en);
	
		//GGF3.3
		$this->pane->named('browseStatus')->setValue('[ '.($this->myModel["_page"]).' | '.
		((count($this->myModel['_where']->clauses()) ? 'filtered' : 'unfiltered')).' ]');
	}

	/**
	 * renders the main frameset
	 *
	 * needs to be overwritten for own window content
	 *
	 * no <body> section in here if $this->pane is a frameset
	 * @see initWindow
	 */
  	protected function openBody() {
		
		// GGF3.2.2
  		// if the list is large, we need a scrollbar
  		// must be done here, after event-processing
  		// an empty scrollbar object already has been created in initWindow()
		$mc = &windowContext($this->myContextID,0);
  		
  		// scroll pane
  		if ($this->myModel['_pagesize'] == 0) {
			$this->scrollPane->add(new GGFGenericControl("mainfs", '
			<frameset cols="*">  <!-- Frameset-Definition -->
			  <frame src="'.$mc->frameCallbackURL(4).'" name="listArea"  frameborder="0" framespacing="0" border="0">
			  <noframes>
			    Your Web-Browser cannot show frames. Use a modern browser.
			  </noframes>
			</frameset>'));
		} else {
			$this->scrollPane->add(new GGFGenericControl("mainfs", '
			<frameset cols="*,20">  <!-- Frameset-Definition -->
			  <frame src="'.$mc->frameCallbackURL(4).'" name="listArea"  frameborder="0" framespacing="0" border="0">
  			  <frame src="'.$mc->frameCallbackURL(5).'" name="scrollBar"  frameborder="0" framespacing="0" border="0">
			  <noframes>
			    Your Web-Browser cannot show frames. Use a modern browser.
			  </noframes>
			</frameset>'));
		}
		
  		// scrollbar
		if ($this->myModel['_pagesize'] > 0) {
		   //frame5
		   $this->scrollBar->add(new GGFGenericControl("scrollbarfs", '
			<frameset rows="35,*,*,35">  <!-- Frameset-Definition -->
			  <frame src="'.$mc->frameCallbackURL(6).'"    name="top" marginwidth="0" marginheight="0" scrolling="no" frameborder="0" framespacing="0" border="0">
			  <frame src="'.$mc->frameCallbackURL(7).'"     name="up" marginwidth="0" marginheight="0" scrolling="no" frameborder="0" framespacing="0" border="0">
			  <frame src="'.$mc->frameCallbackURL(8).'"   name="down" marginwidth="0" marginheight="0" scrolling="no" frameborder="0" framespacing="0" border="0">
			  <frame src="'.$mc->frameCallbackURL(9).'" name="bottom" marginwidth="0" marginheight="0" scrolling="no" frameborder="0" framespacing="0" border="0">
			  <noframes>
			    Your Web-Browser cannot show frames. Use a modern browser.
			  </noframes>
			</html>'));
		   $this->pane->addFrame(5,$this->scrollBar);
		}

  		
  		
 		echo $this->pane->render();
 	}
}
?>