<?PHP
/**
 * This class models the execution-context of a window or
 * dialog. This is the most important abstraction for persistence
 * handling of the dialog flow.
 * 
 * @package GGF
 * @version 4.0
 * @since 1.0
 * @author Gerald Zincke, Austria
 * @copyright 2005-2011 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.html#top
 */
class GGFContext {

   public $contextID;
   public $parentContextID;
   public $model;
   public $modelUndo;
   public $modelRedo;
   public $modelRedoIndicator;
   public $className;
   public $isValid;
   public $result;
   public $continue;
   public $frames = array();
   /*
    * @var array string registry of callback URLs. used to protect system against tampering
    */
   protected $cbURLreg = array();
   

	/**
	 * 
	 * This creates a new window/dialog execution context
	 * Do not call this directly. 
	 * Use dialogContextWithFrom() to create a context for a sub-dialog
	 * and windowContext() to create context for a window
	 * 
	 * @param integer $newcid - id of th context
	 * @param integer $pcid - parent context id
	 * @param array $mo - model object
	 * @param string $cn - class name of window/dialog
	 */
    function __construct($newcid, $pcid, $mo, $cn) {
		//global $contextID;
	   	//global $parentContextID;
	   	//global $model;
	   	//global $className;
	   	//global $isValid;
	   	global $traceLevel;
	
		$this->contextID = $newcid;
		$this->parentContextID = $pcid;
		if ($traceLevel>=2) { 
			error_log_adv ("new context ".$newcid." for ".$cn." created from ".$pcid); 
			if ($cn == "GGFMainWindow") {
				$trace = "";
				$calls = debug_backtrace();
				foreach($calls as $c) {
					$trace = $trace. " -from ".$c["file"]." function:".$c["function"]." line:".$c["line"];
				}
				error_log_adv ($trace);		
			}
		}
		$this->model = $mo;	   
		$this->modelUndo = array();	   
		$this->modelRedo = array();	   
		$this->modelRedoIndicator = 0;	   
		$this->className = $cn;
		$this->isValid = TRUE;
		$this->result = TRUE;
		$this->continue = "";
		return $this;
	}
	
	/**
	 * 
	 * For future use
	 */   
   	public function resetUndoStack() {
   		$this->modelUndo = array();	   
		$this->modelRedo = array();	   
		$this->modelRedoIndicator = 0;	   
   	}
   
   	/**
	 * 
	 * For future use
	 */   
   	public function pushUndo() {
   		// this is called from process-request
   		if (count($this->modelUndo)>20) {
   			array_shift($this->modelUndo);
   		}
   		array_push($this->modelUndo, unserialize(serialize($this->model)));
   		$this->modelRedoIndicator--;		
   		//error_log_adv('pushUndo: ind='.$this->modelRedoIndicator);
   		
   		// This is called at every process Request. A non-undo invalidates the redo stack
   		if ($this->modelRedoIndicator<0) {
   			$this->modelRedo = array();
   			$this->modelRedoIndicator=0;
   			//error_log_adv('pushUndo: ind=0');	
   		}
	}
   
	/**
	 * 
	 * For future use
	 */   
   	public function undo() {
		//error_log_adv('undo');
   		if (count($this->modelUndo)>1) {
			array_pop($this->modelUndo);
			if ($undo = array_pop($this->modelUndo)) {
				$this->pushRedo(); // remember for possible later redo
				$this->modelRedoIndicator++;
				$this->model = $undo;
				$this->save();
			}
		}
	}
	
	/**
	 * 
	 * For future use
	 */   
   	public function pushRedo() {
   		// this is called from the undo operation
   		
   		if (count($this->modelRedo)>20) {
   			array_shift($this->modelRedo);
   		}
   				
   		if ($this->modelRedoIndicator<1) {
   			$this->modelRedo = array();
   			
   			array_push($this->modelRedo, unserialize(serialize($this->model)));
   			$this->modelRedoIndicator=1;
			//error_log_adv('pushRedo: ind=1');
   		
   		} else {
   			array_push($this->modelRedo, unserialize(serialize($this->model)));
   			$this->modelRedoIndicator++;	
			//error_log_adv('pushRedo-else: ind='.$this->modelRedoIndicator);
   		} 
	}
   
   	/**
	 * 
	 * For future use
	 */   
   	public function redo() {
		if (count($this->modelRedo)>0) {
			if ($redo = array_pop($this->modelRedo)) {
				$this->model = $redo;
   				//$this->modelRedoIndicator--;		
				//error_log_adv('redo. ind='.$this->modelRedoIndicator);
   				$this->save();
			}
		}
	}

	/**
	 * 
	 * For future use
	 */   
   	public function canRedo() {
		//error_log_adv('canRedo');
		return ((count($this->modelRedo)>0) && ($this->modelRedoIndicator>=0));
	}
	
	/**
	 * 
	 * For future use
	 */   
   	public function canUndo() {
		//error_log_adv('canUndo');
		return ((count($this->modelUndo))>1); // there is one additional push done before this is evaluated 
	}
   
	/**
	 * 
	 * this allows to access the context of the parent window or -dialog
	 * note: this should not be used to access the parent's model
	 * (a dialog should work on its own model object, supplied by the caller)
   	 */
	public function parentContext() {
   		return windowContext($this->parentContextID, 0);
	}

	/**
	 * 
	 * Set the name of a function in the parent class that will be 
	 * called when the sub dialog has been closed
	 * @param string $ccb
	 */
	public function continueCallback($ccb) {
   		$this->continue = $ccb;
   		$this->save();
	}

	/**
	 * 
	 * Delivers the complete URL for invoking a callback function
	 */
	function callbackURL() {
		$url = "./GGFDispatch.php?_classname=".$this->className."&_contextID=".$this->contextID
		.'&'.session_name()."=".session_id();
		$this->regCallbackURL(substr($url,2));
		return $url;
	}
   
	/**
	 * 
	 * Registers an URL to be a valid callback
	 * internal use only
	 * 
     * @param string $url
     */
	function regCallbackURL($url) {
		global $traceLevel;
		
		if (is_string($url) && ($url>"")) {
			if (in_array($url, $this->cbURLreg)==FALSE) {
   				$this->cbURLreg[] = $url;
   				if ($traceLevel>=2) { 
					error_log_adv("Info.  Context ".$this->contextID." add URL: ".$url);
   				}
			}
		} else {
			if ($traceLevel>=1) { 
				xdebug_print_function_stack( 'Invalid Url.' );
			}
		}
	}
	
	/**
	 * 
	 * internal use only
	 */
	public function resetCallbackURLreg(){
		global $traceLevel;
		
		$this->cbURLreg = array();
		if ($traceLevel>=2) { 
			error_log_adv("Info.  Context ".$this->contextID." reset cbURLreg");
		}
	}
	/**
    * 
    * Called from process request. checks if the url has been registered in the session.
    * if not, the request will be terminated with a short message.
    * requires ini_set("session.use_trans_sid", 0); (PHP Session ID nicht automatisch an relative links anhngen)
    * 
    * @param string $url
    */	
	public function checkCallbackURL($url) {
   		global $traceLevel;
		
   		if (in_array($url, $this->cbURLreg)) {
   			return TRUE;
   		} else {
   			
			if ($traceLevel>=1) { 
	   			if ($traceLevel>=2) { 
		   			error_log_adv('Error. Context '.$this->contextID.' URL '.$url.' not registered. See: ');
		   			foreach ($this->cbURLreg as $u) {
		   				error_log_adv('   '.$u);
		   			}
				} else {
					error_log_adv('Error. Context '.$this->contextID.' URL '.$url.' not registered. ');
				}
			}
   			echo "Invalid client request rejected. Go back.";
   			exit;
   		}
	}

	/**
	 * 
	 * Delivers the complete URL for invoking a frame
	 */
	function frameCallbackURL($frameNr) {
		$url = $this->callbackURL().'&_frame='.$frameNr ;
		$this->regCallbackURL(substr($url,2));
		return $url;
	}
   
	/**
	 * 
	 * Provides an URL that allows to open a sub dialog or window
	 * directly without using the main window of the application.
	 * Note: This should only be used for applications that implement
	 * browser based authentication. See GGFauthenticate.
	 */
	public function directAccessURL($objectKey) {
		//create an URL to the dispatcher
		global $http_type;
		
		$pos = strpos($_SERVER["REQUEST_URI"],"/GGFDispatch.php");
		$dir = substr($_SERVER["REQUEST_URI"],0,$pos);
		return $http_type.'://'.
		$_SERVER["SERVER_NAME"].':'.
		$_SERVER["SERVER_PORT"].$dir.
		"/GGFDispatch.php?_classname=".$this->className.
		"&_contextID=999999999&".$objectKey;
		// note: this url corresponds to the context-generation code in the constructor
	}
	
	/**
	 * 
	 * Declare the context invalid (when closing the window/dialog)
	 * Do not call this directly. See: GGFControlWindow->eventClose()
	 */
	function invalidate() {
		global $traceLevel;
	   	// don't do this: $this->model = "";
	   	$this->isValid = FALSE;
	   	//echo $this->callbackFilename();
	   	$frameNr = 1;
	   	// remove iframe contents
	   	while (isset($this->frames[$frameNr])) {
	   		unset($this->frames[$frameNr]);
	   		$frameNr++;
	   	}
	   	if ($traceLevel>=2) { 
			error_log_adv("Info.  Context ".$this->contextID." now is invalidated ");
	   	}   	
	   	$this->resetCallbackURLreg();
	}
   
	/**
	 * 
	 * Make sure that the context is stored in the persistent PHP Session object.
	 */
	public function save() {
   		global $traceLevel;
		
   		//if (is_array($this->model[1]) && is_array($this->model[1]["autofocus"])) {
   		//	xdebug_print_function_stack( 'Context save.Invalid model autofocus.' );		
   		//}
   	   $cs = $_SESSION["_cstack"];
	   $cs[$this->contextID] = $this;
	   if ($traceLevel>=2) { 
		   error_log_adv("saving context:".$this->contextID." with page=".$this->model["_page"]);
	   }
	   $_SESSION["_cstack"] = $cs;	   
	}
	
	/**
	 * 
	 * Open a sub-dialog in the same browser tab or window.
	 * Note: this function will not return (exits the current script.)
	 * @param string $dcn - name of the dialog class
	 * @param array $mod - model object for dialog
	 * @param string $cont - optional: name of callback function in parent class that will be called when the sub dialog is closed
	 */
	public function openDialogOn($dcn, $mod, $cont=""  ) { 
  		global $appname;
  		global $http_type;
		$this->save();
   		$newContext = &dialogContextWithFrom(0, $appname, $this->contextID,$dcn); // create a new context and set it active

		$newContext->model = $mod;
		$this->continue = $cont;
		$this->save();
		
		$newContext->save();
		header("Expires: Mon, 26 Jul 1997 05:00:00 GMT");    		// Datum aus Vergangenheit
		header("Last-Modified: " . gmdate("D, d M Y H:i:s") . " GMT"); 	// immer gendert
		header("Cache-Control: no-store, no-cache, must-revalidate");  	// HTTP/1.1
		header("Cache-Control: post-check=0, pre-check=0", false);
		header("Pragma: no-cache");                          		// HTTP/1.0
		header("Location: $http_type://".$_SERVER['HTTP_HOST']
                      .dirname($_SERVER['PHP_SELF'])
                      ."/".$newContext->callbackURL()
                      );
		exit; 	
	}
   
	/**
	 * 
	 * Open a new window opening a seperate browser tab or window.
	 * Note: this function will not return (exits the current script.)
	 * @param string $wcn - name of the window class
	 * @param array $mod - model object for dialog
	 */
	public function openWindowOn($wcn, $mod  ) { // prepare context switch and open a window $wcn on model object $mod
  		global $appname;
  		global $http_type;
   		$newContext = &windowContext(0, $wcn); // create a new context and set it active
		$newContext->model = $mod;
		$newContext->save();
		header("Expires: Mon, 26 Jul 1997 05:00:00 GMT");    		// Datum aus Vergangenheit
		header("Last-Modified: " . gmdate("D, d M Y H:i:s") . " GMT"); 	// immer gendert
		header("Cache-Control: no-store, no-cache, must-revalidate");  	// HTTP/1.1
		header("Cache-Control: post-check=0, pre-check=0", false);
		//header('Target="_blank"');					// does not work this way
		header("Pragma: no-cache");                          		// HTTP/1.0
		header("Location: $http_type://".$_SERVER['HTTP_HOST']
                      .dirname($_SERVER['PHP_SELF'])
                      ."/".$newContext->callbackURL()
		);
		exit; 	
	}
	/**
	 * 
	 * This switches back to the parent context.
	 * Do not call this directly. See: GGFControlDioalog->eventClose()
	 * Note: this function will not return (exits the current script) except
	 * when there is no parent context found.
	 * @param boolean $ok - tells the parent window if it should evaluate the model of the sub dialog
	 */
	public function returnResult($ok) { // re-establish parent-context
	    global $http_type;
	   	$this->result = $ok; // should be true or false, 1, 2 or 0 for yes/no/cancel dialogs
	   	$this->save();
	   	if ($this->parentContextID==0) { // is root
	   		return 0;
	   	}
		$parentContext = &dialogContextWithFrom($this->parentContextID,0,0,0); 
	   	$this->invalidate();
	   	$this->save();
		$url =$parentContext->callbackURL().
				'&_callback='.          $parentContext->continue.
				'&_returnedFrom='.      $this->contextID	;
		
		$parentContext->resetCallbackURLreg(); // beim return wird parent neu gerendert. Erlaubte callbacks knnten vom Ergebnis des Subdialog abhngen
		$parentContext->regCallbackURL(substr($url,2)); // das ist gleich einmal erlaubt
		$parentContext->save();
		
	   	$newHeader = "Location: $http_type://".$_SERVER['HTTP_HOST']
	                      	.dirname($_SERVER['PHP_SELF'])
	                      	."/".$url;
	   	// echo "GGFContext returnResult-header:".$newHeader.
	   	header($newHeader);
		exit;
	}
	/**
	 * 
	 * @deprecated
	 */
	function frameFileName($frameNr) {
		// provide the URL used for processing callbacks
		// currently (4.0) not used anymore
		$this->model["_frame".$frameNr] =  "./frame_".session_id()."_".$this->contextID."_".$frameNr;
		$this->save();
		return $this->model["_frame".$frameNr];
	}
}   
?>