<?PHP

/**
 * A class to implement a dialog (form input) window.
 * The value of $this->ok will be returned in the context as the result to
 * the parent window.
 * 
 * @todo 
 * @package GGF
 * @version 4.0
 * @since 2.0
 * @author Gerald Zincke, Austria
 * @copyright 2005,2011 Gerald Zincke
 * @license http://www.reinhtml.eu/ggf/license.html#top
 */
class GGFControlDialog extends GGFControlWindow {
	// assume the 
	// model[0] contains the the key of the model object
	// model[1] the db row
	
	protected $ok;
	protected $ETypeName = ""; // the name of the entitytype of the model object 
	
	// Sample code: Trim whitespace (including line breaks) at the start and the end of the string
	//	preg_replace("\A\s+|\s+\z", "", $text);

	// Define matching patterns for input classes
	// Note: control input-classes are checked by the processRequest() function of entry fields.
	static $inputPattern = array(
		//0 any text
		"/./", 
		//1 any text without angle brackets or single or double quotes
		"#[<>'\"]#", 
		//2 alphanumerical text with no special characters except underscores and blanks
		"#[^a-zA-Z0-9_ ]#", 
		//3 single word
		"#[^a-zA-Z]#", 
		//4 variable name in most computer languages
		"#(^[^a-zA-Z]|[^a-zA-Z0-9_])#",

		//5 decimal number with or without signor decimal comma
		"/^[-+]?\d*(\,)?\d*$/",
		//6 decimal number with or without sign or decimal point
		"/^[-+]?\d*(\.)?\d*$/",
	
		//7 Date d.m.y and dd.mm.yyyy
		'/^([0-9]|[012][0-9]|3[01])\.(0?[1-9]|1[012])\.(19|20)?[0-9]{2}$/',
		//8 Date m/d/y and mm/dd/yyyy
		'~^(0?[1-9]\/|1[012]\/)([0-9]\/|[012][0-9]\/|3[01]\/)(19|20)?[0-9]{2}$~',
		//9 Date yyyy-mm-dd, 1900-01-01 through 2099-12-31. Matches invalid dates such as February 31st
		'/^(19|20)[0-9]{2}[-](0[1-9]|1[012])[-](0[1-9]|[12][0-9]|3[01])$/',
	
		//10 time. Hours and minutes (hh:mm), 24-hour clock
		"/^(2[0-3]|[01]?[0-9]):([0-5]?[0-9])$/", 
		//11 time. Hours, minutes and seconds (hh:mm:ss), 24-hour clock
		"/^(2[0-3]|[01]?[0-9]):([0-5]?[0-9]):([0-5]?[0-9])$/", 
		//12 time. Hours and minutes (hh:mm AM), 12-hour clock:
		"/^(1[0-2]|0?[1-9]):([0-5]?[0-9] [aApP][mM])$/",
		//13 time. Hours, minutes and seconds (hh:mm:ss Pm), 12-hour clock" 
		"/^(1[0-2]|0?[1-9]):([0-5]?[0-9]):([0-5]?[0-9] [aApP][mM])$/", 
		
		//14 Email address
		"/^[a-zA-Z0-9-_.]+@[a-zA-Z0-9-_.]+\.[a-zA-Z]{2,8}$/",
	
		//15 URL see here: http://regexlib.com/Search.aspx
		'/^((https?|ftp)\:\/\/)?([a-z0-9+!*(),;?&=$_.-]+(\:[a-z0-9+!*(),;?&=$_.-]+)?@)?([a-z0-9-.]*)\.([a-z]{2,3})(\:[0-9]{2,5})?(\/([a-z0-9+$_-]\.?)+)*\/?(\?[a-z+&$_.-][a-z0-9;:@&%=+\/$_.-]*)?(#[a-z_.-][a-z0-9+$_.-]*)?$/',
		
		//16 Password complexity
		'/^(?=[-_a-zA-Z0-9]*?[A-Z])(?=[-_a-zA-Z0-9]*?[a-z])(?=[-_a-zA-Z0-9]*?[0-9])[-_a-zA-Z0-9]{6,}$/',
	
		//17 HEX color code
		'/^#[A-F0-9]{6}$/i',
		//18 a text for phone number: plus, numbers, brackets, blanks, minus, sharp and star  
		"#[^\+0-9 \#\*\-]#" 
		
	);
			
	// for internal use only
	static $inputPatternText = array(
		"any string, no check (leading and trailing blanks are not trimmed)", 
		"any string without angle brackets,single or double quotes",
		"english or german letters, numbers,underscores, hyphens or blanks",
		"single word with letters only",
		"variable name (as in most computer languages starting with a letter, then alphanumeric)",
	
		"decimal number like -123,45 with or without decimal comma, with or without sign",
		"decimal number like -123.45 with or without decimal point, with or without sign",
		
		"date d.m.yy and dd.mm.yyyy, 1.1.99 through 31.12.2099. ",
		"date m/d/y and mm/dd/yyyy, 1/1/99 through 12-31-2099. ",
		"date yyyy-mm-dd, 1900-01-01 through 2099-12-31.",
		
		"time hh:mm with 24-hour clock.",
		"time hh:mm:ss with 24-hour clock.",
		"time hh:mm with 12-hour clock (AM/PM).",
		"time hh:mm:ss with 12-hour clock (AM/PM).",
		
		"valid email address",
		"Absolute URL. http://domain:port/file, https: or ftp:",
	 	"password, consisting of 6 or more characters and contain at least one upper case letter, one lower case letter and one digit",
	
		"HTML color code from #000000 (black) to #FFFFFF (white).",
		"Phone number characters only"
 	);
	 
		
	/**
	 * 
	 * Dialog constructor. Call parent if you override it. 
	 * @param integer $contextID
	 */
 	function __construct($contextID) {
       	$this->myContainerClass="GGFDialogFormPane";
       	parent::__construct($contextID);
		$this->ok = FALSE;
   	}
		
   	/**
	 * Internal use only.
	 * Override this, if your dialog should have a different CSS class for  the HTML body
	 */	 		
 	public function bodyClass() { 
		return 'GGFControlDialog'; //for CSS
	}
   		
	/**
	 * 
	 * for future use.
	 */
	protected function modelObjectPrimaryKeyValuesString() {
		// note: dialogs should allow taking notes tied to the displayed model-object
		// for that the eventAddNotes function needs a string identifying the object
		
		global $ERModel;	
		if ($this->ETypeName == "") {
			return ""; // no object notes possible
		} else {
			$et = $ERModel->entityTypeNamed($this->ETypeName);
			if ($et == NULL) {	
				return "";
			}
			return $et->primaryKeyValuesString($this->modelObjectRow());
		}
		// where $et is the entitytype of the modelobject and $modelObjectRow gives its DB row
	}
	
	/**
	 * 
	 * internal use only
	 */
	protected function modelObjectRow() {
		return $this->myModel[1];
	}
	
	/**
	 * 
	 * This interface should return true, if processing of data
	 * can be done and active controls should be enabled.
	 * @return boolean
	 */
   	public function dataPlausible() {
		// if true, OK can be clicked
		return $this->isLoggedOn();
	}

	/**
	 * 
	 * Returns an URL that
	 * allows to open the window or dialog without going via the main window
	 * Note: This should only be used for applications that implement
	 * browser based authentication. See GGFauthenticate and GGFContext.
	 * @return string
	 */		
	public function directAccessURL() {
		//create an URL to the dispatcher
		if (is_string($this->myModel[0])) {
			$c = &windowContext($this->myContextID, "");
			return $c->directAccessURL("pkvalue=".$this->myModel[0]);
		// note: this url corresponds to the context-generation code in the constructor
		} else {
			return "";
		}
	}
	
	//----- Window processing-------------------------------------
	//	$this->initModel($mc); // build/initialize myModel
 	//	$this->initWindow($mc); // fill pane with controls
 	//	$this->processRequest($mc); // read data from $_REQUEST into controls
 	//	$this->processCallbacks($mc); // process callBacks if any
 	//	$this->fillControls($mc);
 	//	$this->openHTML();
	
	/**
	 * 
	 * This is where the appearance of the dialog is to be defined.
	 * Called by this->open().
	 * Derived classes must override this functions, but call parent.
	 * @param GGFContext $mc
	 */
	protected function initWindow($mc) {
		parent::initWindow($mc); // has already an empty menu-pane
		// in concrete classes overwrite this method and call parent
		// add controls there
		//GGF3.1.2
		$form = new GGFControlPane($this,"_dialog");
		$form->newRow(new GGFControlPane($this,"_mainform"));
		$buttonarea = new GGFControlPane($this,"_buttonarea");
		$buttonarea->add(new GGFPushbutton("_OK",     "OK",     "eventOK",    "dataPlausible", 60, "save data in database", 'width:80px; overflow:hidden'));
		$buttonarea->newSpace(1);
		$buttonarea->add(new GGFPushbutton("_Cancel", "Cancel", "eventClose", "", 60, "do not save data in database"));
		$form->closeTable();
		$form->newRow($buttonarea, "","   width:100%; ", " width:98%; ");
		//GGF3.0
		$form->newCell($this->systemTray($mc)," text-align:right; ");
		$this->pane->add($form);
	}		

	/**
	 * 
	 * Read the data from entry-field controls and selector-controls into
	 * the model object. This default implementation assumes that
	 * $this->myModel[1] contains a key->value array with a database row where column names are keys and
	 * the names of the entry fields are the same as the column names. It is recommended to overide this 
	 * to read selected values from selection controls.
	 * Note: control input-classes are checked by the processRequest() function of entry fields.
	 * No need to to it here again
	 * @param GGFContext $mc
	 */
	protected function readControls($mc) {
		// this is called before processing callbacks, transfers the 
		// value of non-readonly entry controls into 
		// $mc->model and $this->myModel[1]
		global $errorStack;

		foreach($this->myModel[1] as $key => $value) {
			// $errorStack->pushMsg(print_r($key, TRUE));
			if (!($this->pane->named($key)==0)) {
				$c = $this->pane->named($key);
				if (!$c->isReadonly()) {
					if ($c->isEntry()) {
						//$errorStack->pushMsg($key."=".$this->pane->named($key)->value());
						$this->myModel[1][$key] = $this->pane->named($key)->value();
					} elseif (is_a($c, 'GGFSelectionControl')) {
						if (strcmp($c->selectionsType(),"array")==0) {
							$ar = $this->pane->named($key)->getSelection();
							$this->myModel[1][$key] = $ar[0]; 
						} else {
							$this->myModel[1][$key] = $this->pane->named($key)->getSelection();
						}
						
					}
				}
			}
		}
		$mc->model = $this->myModel;
		$mc->save();
	}
	
	/**
	 * Interface to save all changes made in an update dialog (to a database for instance). 
	 * Before updating the storage a validation of data is done.
	 * Called from eventOK()
	 */
	protected function saveModifications() {
		global $db; // This global object of class GGFDatabase provides an open DB connection. See also GGFSetup.
		global $errorStack;
		
		// verification of data before Update. If verification fails a message is posted in $errorStack
		if ($this->validateData()) {
			$this->ok = TRUE;		// success = true;
		}
	}

	/**
	 * checks input validity of data 
	 * Override this in derived classes, call parent at the end.
	 * Note: control input-classes are checked by the processRequest() function of entry fields.
	 * Called from saveModifications()
	 *
	 */
	protected function validateData() {
		// add verification clauses here and push message to $errorStack in case of failure.
		// Return success.
		global $errorStack;		
		
		if (!$errorStack->isEmpty()) {
			return $this->eventDisplayError();
		}
		return true;
	}
	
	//---events------------------------------------------------------
	
	/**
	 * 
	 * Callback function to finish input, do data processing, save results and close the dialog.
	 * Usually triggered by OK button. (Cancel calls eventClose)
	 */
	protected function eventOK() {
		// here we want to come wenn the OK-button is clicked
		// tell the difference between buttons by their name ~ menuitem[0}
		// you may set the $ok variable here after processing input
		$this->saveModifications();
		if ($this->ok) {
			$this->eventClose();
		}
	}
	
	/**
	 * 
	 * Callback function to close the dialog without processing input data.
	 * Usually triggered by Cancel button. (OK calls first eventOK and then this function)
	 * If the dialog was opened from a window, processing will be transfered back to it.
	 */
	protected function eventClose() {
   		// as we are a dialog, there is a parent context we have to continue with
   		// go back in dialog stack or to bye window
   		global $appname;
   		global $goodbyefile;
   		global $errorStack;
   		
   		if (!$this->ok) { // dialog has been cancelled
   			// ignore messages left in stack
   			$errorStack->popAll();
   		}
   		
   		$myContext = &dialogContextWithFrom($this->myContextID,0,0,0);
   		// error_log_adv('GGFDialog->eventClose:'.print_r($myContext, true));
   		if ($myContext->returnResult($this->ok)==0) {
   			// no parent to return (may have been opened by direct access)
	   		$myContext->invalidate();
   			header('Location: '.$goodbyefile);
			exit;
   		}   		
	}

}
?>